#!/usr/bin/perl
#
# usage:  HexSplitter.pl HEXFILE
#
# This script splits a single HEXFILE outputed by HexToDownload.pl
# into multiple hex files so that each output file only contains a
# range of addresses within 64K.
#
# Output files are created, HEXFILE.0, HEXFILE.1, ..., that contain
# the split data.

# Check arguments
if ($#ARGV < 0) {
  print STDERR "usage: HexSplitter.pl HEXFILE\n";
  exit 1;
}

# Dump the input hex file
my $progHexFilename = shift;
open (HANDLE, "$progHexFilename")
    || die "cannot open $progHexFilename for reading.";

my @records = ( );

while (my @readData = hexReadData()) {
    push(@records, \@readData);
}

# Close the input hex file
close(HANDLE);

# Sort the records numerically on the address field
@records = sort {@$a[0] <=> @$b[0]} @records;

$hexLastAddr = 0;

# Initialize the bankNum
# This variable is used as the extension on the output file.
$bankNum = 0;
$firstRun = 1;

while (my $outRecordRef = shift(@records)) {
    my @outRecord = @$outRecordRef;
    my $addr = shift(@outRecord);

    if ( (($addr & 0xffff0000) != ($hexLastAddr & 0xffff0000)) || $firstRun) {
        # If OUTHANDLE is invalid (i.e. this first time through)
        # the following two statements won't do anything nor will they
        # product any errors.
        hexEofRecord();
        close (OUTHANDLE);

        # Start the new bank
        open (OUTHANDLE, ">$progHexFilename.$bankNum")
            || die "cannot open $progHexFilename.$bankNum for writing.";
        ++$bankNum;
        $firstRun = 0;
        $hexLastAddr = $addr;
    }

    hexDataRecord($addr, @outRecord);
}

# Finish up the last bank
hexEofRecord();
close(OUTHANDLE);

# EOF
exit 0;


#
# HEX file utilities
#
$hexLastAddr = 0;
$hexHi       = 0;
$hexLo       = 0;

# Reset the hex record state variables
sub hexReset() {
    $hexHi = 0;
    $hexLo = 0;
}

# Read the next hex record
sub hexReadData() {
    while (<HANDLE>) {
       # Extended linear address
       if (/^:02000004([0-9a-f]{4})/i) {
          $hexHi = $1;
       }
       
       # Data Record
       elsif (/:([0-9a-f]{2})([0-9a-f]{4})00([0-9a-f]+)([0-9a-f]{2})/i) {
           $length = hex($1);
           $hexLo  = $2;
           @bytes  = split(//, $3);
           
           @ret = (hex("$hexHi$hexLo"));
           for ($i = 0; $i <= $#bytes; $i += 2) {
               push(@ret, hex("$bytes[$i]$bytes[$i+1]"));
           }
           
           return @ret;
       }
       
       # End of File
       elsif (/^:00000001ff/i) {
           return ();
       }
    }
}

# Write a hex data record
sub hexDataRecord($@) {
    my $addr         = shift;
    my @bytes        = @_;

    # If the high word of the address has changed, output an
    # extended linear address record
    if (($addr & 0xffff0000) != ($hexLastAddr & 0xffff0000)) {
        $addrHi = sprintf("%04X", ($addr >> 16) & 0xffff);
        $output  = ":02000004$addrHi";
        $output .= checksum($output);
        print OUTHANDLE "$output\n";
    }
    
    # Output the data record
    $output = sprintf(":%02X%04X00", $#bytes+1, ($addr & 0xffff));
    for ($i = 0; $i <= $#bytes; ++$i) {
        $output .= sprintf("%02X", $bytes[$i]);
    }
    $output .= checksum($output);
    print OUTHANDLE "$output\n";
    
    $hexLastAddr = $addr;
}

# Write a hex EOF record
sub hexEofRecord() {
    printf OUTHANDLE ":00000001FF\n";
}

# Checksum subroutine
sub checksum($) {
    my $record = shift;
    $record =~ /:([0-9a-f]+)/i;
    my @bytes = split(//, $1);

    my $sum = 0;
    for (my $i=0; $i <= $#bytes; $i += 2) {
        my $nybble = hex("$bytes[$i]$bytes[$i+1]");
        $sum += $nybble;
    }
    
    return sprintf("%02x", ((-$sum) & 0xff));
}
