#!/usr/bin/perl
#
# usage:  HexToDownloader.pl DATFILE
#
# This script processes the set of hex files into a format
# readable by the hex downloading program.  The final output
# HEX file contains both the downloader program and the processed
# data files.
#

# Check arguments
if ($#ARGV != 0) {
    print STDERR "usage: HexToDownloader.pl DATFILE\n";
    exit 1;
}

# Read the DAT file
open (HANDLE, "$ARGV[0]")
    || die "cannot open $ARGV[0] for reading.";

while (<HANDLE>) {
    $dat .= $_;
}
eval($dat);

close (HANDLE);


# Dump the program hex file
my $progHexFilename = $DOWNLOADER;
open (HANDLE, "$progHexFilename")
    || die "cannot open $progHexFilename for reading.";
    
while (my @readData = hexReadData()) {
    my $addr = shift(@readData);
    hexDataRecord($addr, @readData);
}

close(HANDLE);


# Read each data file, dumping their contents as address/data blocks
# into the output hex file.
$currHexAddr   = 0x10400;
$dataIndexAddr = 0x10000;
@dataIndex     = ();
while (my $dataHexFilename = shift(@IMAGES)) {
    open (HANDLE, "$dataHexFilename")
        || die "cannot open $dataHexFilename for reading.";

    $addr = $currHexAddr - $dataIndexAddr;
    
    push(@dataIndex, ($addr >> 24) & 0xff);
    push(@dataIndex, ($addr >> 16) & 0xff);
    push(@dataIndex, ($addr >>  8) & 0xff);
    push(@dataIndex, ($addr >>  0) & 0xff);

    hexReset();
    @writeData = (memtype($MEMORY_TYPE),
                  ($MEMORY_CAP >> 24) & 0xff,
                  ($MEMORY_CAP >> 16) & 0xff,
                  ($MEMORY_CAP >>  8) & 0xff,
                  ($MEMORY_CAP >>  0) & 0xff);
    hexDataRecord($currHexAddr, @writeData);
    $currHexAddr += $#writeData +1;
    
    @writeData = zeropad($MEMORY_ALG, 31);
    hexDataRecord($currHexAddr, @writeData);
    $currHexAddr += $#writeData +1;

    while (my @readData = hexReadData()) {
        $addr = shift(@readData);
        @writeData = (($addr >> 24) & 0xff,
                      ($addr >> 16) & 0xff,
                      ($addr >>  8) & 0xff,
                      ($addr >>  0) & 0xff,
                      $#readData +1,
                      @readData);

        hexDataRecord($currHexAddr, @writeData);
        $currHexAddr += $#writeData +1;
    }

    # Signal the end of this block
    hexDataRecord($currHexAddr, 0xff, 0xff, 0xff, 0xff);
    $currHexAddr += 4;
    
    close(HANDLE);
}


# Dump the index
hexDataRecord($dataIndexAddr, @dataIndex, 0xff, 0xff, 0xff, 0xff);

# EOF
hexEofRecord();
exit 0;


#
# HEX file utilities
#
$hexLastAddr = 0;
$hexHi       = 0;
$hexLo       = 0;

# Reset the hex record state variables
sub hexReset() {
    $hexHi = 0;
    $hexLo = 0;
}

# Read the next hex record
sub hexReadData() {
    while (<HANDLE>) {
       # Extended linear address
       if (/^:02000004([0-9a-f]{4})/i) {
          $hexHi = $1;
       }
       
       # Data Record
       elsif (/:([0-9a-f]{2})([0-9a-f]{4})00([0-9a-f]+)([0-9a-f]{2})/i) {
           $length = hex($1);
           $hexLo  = $2;
           @bytes  = split(//, $3);
           
           @ret = (hex("$hexHi$hexLo"));
           for ($i = 0; $i <= $#bytes; $i += 2) {
               push(@ret, hex("$bytes[$i]$bytes[$i+1]"));
           }
           
           return @ret;
       }
       
       # End of File
       elsif (/^:00000001ff/i) {
           return ();
       }
    }
}

# Write a hex data record
sub hexDataRecord($@) {
    my $addr   = shift;
    my @bytes  = @_;

    # If the high word of the address has changed, output an
    # extended linear address record
    if (($addr & 0xffff0000) != ($hexLastAddr & 0xffff0000)) {
        $addrHi = sprintf("%04X", ($addr >> 16) & 0xffff);
        $output  = ":02000004$addrHi";
        $output .= checksum($output);
        print "$output\n";
    }
    
    # Output the data record
    $output = sprintf(":%02X%04X00", $#bytes+1, ($addr & 0xffff));
    for ($i = 0; $i <= $#bytes; ++$i) {
        $output .= sprintf("%02X", $bytes[$i]);
    }
    $output .= checksum($output);
    print "$output\n";
    
    $hexLastAddr = $addr;
}

# Write a hex EOF record
sub hexEofRecord() {
    printf ":00000001FF\n";
}

# Checksum subroutine
sub checksum($) {
    my $record = shift;
    $record =~ /:([0-9a-f]+)/i;
    my @bytes = split(//, $1);

    my $sum = 0;
    for (my $i=0; $i <= $#bytes; $i += 2) {
        my $nybble = hex("$bytes[$i]$bytes[$i+1]");
        $sum += $nybble;
    }
    
    return sprintf("%02x", ((-$sum) & 0xff));
}

# Memory type
sub memtype($) {
    my $name = shift;
    return 1 if ($name =~ /^EXT_SRAM$/);
    return 2 if ($name =~ /^FLASH$/);
    return 3 if ($name =~ /^SEEPROM$/);
    return 0;
}

# Zero-padding function
sub zeropad($$) {
    my $s = shift;
    my $n = shift;
    my @result = split(//, $s);
    for (my $i = 0; $i < $n; ++$i) {
        $result[$i] = ord($result[$i]);
    }
    return @result;
}
