/*=========================================================================
| (c) 2000  Triscend Corporation
|--------------------------------------------------------------------------
| Project : Embedded E5 Jtag Library
|
| File    : te5dl.c
 ========================================================================*/

/*=========================================================================
| INCLUDES
 ========================================================================*/
#include <string.h>
#include "te5dl.h"
#include "te5jtag.h"
#include "te5disp.h"


// Data structure for the programming algorithms
typedef struct {
    data_32 address;
    data_8  numBytes;
    data_8  bytes[16];
} DataRecord;

#include "te5flash.h"
#include "te5sprom.h"

/*=========================================================================
| MACROS
 ========================================================================*/
#define READ_BIG_ENDIAN_32(addr) \
    (((data_32)memRead((addr) +0L) << 24) | \
     ((data_32)memRead((addr) +1L) << 16) | \
     ((data_32)memRead((addr) +2L) <<  8) | \
     ((data_32)memRead((addr) +3L) <<  0))

#define READ_BIG_ENDIAN_16(addr) \
    (((data_16)memRead((addr) +0L) << 8) | \
     ((data_16)memRead((addr) +1L) << 0))

// 1) Call function()
// 2) On error, return
#define _CKERR_VOID(function) \
    { \
        jtagError = JTAG_OK; \
        gDlError = DL_OK; \
        function; \
        if (jtagError != JTAG_OK) gDlError = DL_JTAG_ERROR; \
        if (gDlError  !=   DL_OK) return; \
    }


/*=========================================================================
| CONSTANTS
 ========================================================================*/
#define MAX_PROGRAMS  16

#define DATA_BUFFER_EXP      12      /* Between 7 and 14.   Ex: 2^12 = 4KB */
#define DATA_BUFFER_SIZE     (1 << DATA_BUFFER_EXP)
#define DATA_BUFFER_MASK     (0xffffffffL << DATA_BUFFER_EXP)

#define EXT_MEM_BASE         0x810000L  // EXTMEM + 64KB

// MIU setup addresses
#define REG_MIUMODE          0x020e30L
#define REG_PIOSEL_0         0x020e73L
#define REG_PIOSEL_1         0x020e74L

#define ADDR_EOF             0xffffffffL


/*=========================================================================
| CONSTANTS (PROGRAMMED MEMORY)
 ========================================================================*/
// Registers for mapping the programming algorithm
// Common registers
#define REG_ENAREG          0x020e72L
#define REG_CMAP1_CTL       0x020f06L
#define REG_CMAP2_TAR_0     0x020f08L
#define REG_CMAP2_TAR_1     0x020f09L
#define REG_CMAP2_TAR_2     0x020f0aL
#define REG_CMAP2_SRC       0x020f0bL
#define REG_CMAP2_CTL       0x020f0cL

// Registers for SEEPROM programming
#define REG_DMAP3_CTL       0x020f1bL
#define REG_DMAP5_TAR_0     0x020f85L
#define REG_DMAP5_TAR_1     0x020f86L
#define REG_DMAP5_TAR_2     0x020f87L
#define REG_DMAP5_SRC       0x020f88L
#define REG_DMAP5_CTL       0x020f89L
#define REG_MIUSCTL         0x020e33L

// Define a 2K block size for the serial EEPROM
#define SEEPROM_BLOCK_SIZE  (2 << 10)

// Addresses used by the programming algorithms
#define PROG_COMMAND        0x00010ff0L
#define PROG_CONTROL        0x00010ff1L
#define PROG_STATUS         0x00010ff2L
#define PROG_ADDR           0x00010ff4L
#define PROG_COUNT          0x00010ff8L
#define PROG_DATA           0x00011000L
#define PROG_ABS_START      0x00800000L
#define PROG_DEVICE         0x00010fd0L

#define PROG_CMD_PROGRAM    0x50
#define PROG_CMD_BULK_ERASE 0x42
#define PROG_CMD_SET_DEVICE 0x44

#define PROG_STAT_IDLE      0x49

#define PROG_CTL_READY      0x52
#define PROG_CTL_GO         0x47
#define PROG_CTL_ACK        0x41

// Programming algorithm error codes are in te5dl.h

/*=========================================================================
| EXTERNALS
 ========================================================================*/
// These should be defined by the project's header file
extern data_8 DMAP5_TAR_0;
extern data_8 DMAP5_TAR_1;
extern data_8 DMAP5_TAR_2;
extern data_8 DMAP5_SRC;
extern data_8 DMAP5_CTL;

extern data_8 DMAP3_TAR;
extern data_8 DMAP3_SRC;
extern data_8 DMAP3_CTL;


/*=========================================================================
| STATIC VARIABLES
 ========================================================================*/
static data_32 gProgramAddresses[MAX_PROGRAMS];
static data_8  gNumPrograms;
static data_8  gBytes[2048];

static xdata volatile data_8 gExtMem[DATA_BUFFER_SIZE]  _at_  0x8000;
static data_32               gMemOffset = 0;


/*=========================================================================
| GLOBAL VARIABLES
 ========================================================================*/
data_8 gDlError = DL_OK;


/*=========================================================================
| STATIC PROTOTYPES
 ========================================================================*/
static void   _progWriteMemory (data_32 addr, data_16 len, data_8 bytes[]);
static void   _targetWriteByte (data_32 addr, data_8 value);
static void   _dlProgrammer (MemoryType memoryType, data_8 algorithmName[]);
static void   _issueProgCommand (data_8 command);

static void   memMap (data_32 mem_addr);
static data_8 memRead (data_32 addr);
static void   memWrite (data_32 addr, data_8 value);
static void   setupMIU (data_32 capacityKB);
    

/*=========================================================================
| CONSTRUCTOR
 ========================================================================*/
void dlInitialize ()
{
    int     i;
    data_32 addr;

    // Read the index table of downloadable programs
    for (i = 0, addr = 0; i < MAX_PROGRAMS; ++i, addr += 4) {
        gProgramAddresses[i] = READ_BIG_ENDIAN_32(addr);
        if (gProgramAddresses[i] == ADDR_EOF) break;
    }
    
    gNumPrograms = i;
}


/*=========================================================================
| DESTRUCTOR
 ========================================================================*/
void dlDeinitialize ()
{
    // Do nothing
}


/*=========================================================================
| FUNCTIONS
 ========================================================================*/
// Download the program given by index into the target board's SRAM
void dlDownload (char index)
{
    data_32    sramAddr;
    data_32    addr = 0;
    data_32    alignedAddr = 0;
    data_8     len = 0;
    data_16    i;
    data_32    bytesDownloaded = 0;
    data_32    offset = 0;
    MemoryType memoryType;
    data_32    capacityKB;
    data_8     algorithmName[32];
    
    if (index >= gNumPrograms) {
        gDlError = DL_PROG_INDEX_OUT_OF_RANGE;
        return;
    }
    
    sramAddr = gProgramAddresses[index];
    
    // Load the memory type, capacity, and algorithm name
    memoryType = (MemoryType)memRead(sramAddr +0);
    capacityKB = READ_BIG_ENDIAN_32(sramAddr +1);
    
    for (i = 0; i < 31; ++i) {
        algorithmName[i] = memRead(sramAddr +5 +i);
    }
    algorithmName[i] = '\0';
    sramAddr += 36;

    // Reset the CPSU completely and halt
    _CKERR_VOID( jtagResetCpsu() );
    _CKERR_VOID( jtagResetAndHalt() );
    
    _CKERR_VOID( setupMIU(capacityKB) );

    // Do some setup
    switch (memoryType) {
      case INTERNAL_SRAM:
        offset = INTERNAL_RAM_OFFSET; break;
        
      case EXTERNAL_SRAM:
        offset = EXTERNAL_RAM_OFFSET; break;
        
      case FLASH_MEMORY:
      case SEEPROM_MEMORY:
        // An offset does not need to be applied to the addresses
        // because the programmer takes care of the addressing
        _CKERR_VOID( _dlProgrammer(memoryType, algorithmName) );
        break;
    }

    // Loop through the data
    for (;;) {
        if (memoryType != SEEPROM_MEMORY) {
            // Read the address and data block length
            addr = READ_BIG_ENDIAN_32(sramAddr);
            len  = memRead(sramAddr +4);
            sramAddr += 5;
            
            // Download is complete if the EOF tag is found
            if (addr == ADDR_EOF) break;
            
            // Load the data into the target board
            for (i = 0; i < len; ++i) {
                gBytes[i] = memRead(sramAddr++);
            }

            // Update the displayed variable
            bytesDownloaded += len;
        } else {
            i = 0;
            while (i < SEEPROM_BLOCK_SIZE) {
                if (!len && addr != ADDR_EOF) {
                    // Read the address and data block length
                    addr = READ_BIG_ENDIAN_32(sramAddr);
                    len  = memRead(sramAddr +4);
                    sramAddr += 5;
                }

                // Download is complete if the EOF tag is found
                if (i == 0 && addr == ADDR_EOF) break;
                
                if (i == 0)
                    alignedAddr = addr - (addr % SEEPROM_BLOCK_SIZE);

                while (alignedAddr + i != addr && i < SEEPROM_BLOCK_SIZE) {
                    gBytes[i] = 0xff;
                    ++i;
                }
                
                while (len && i < SEEPROM_BLOCK_SIZE) {
                    gBytes[i] = memRead(sramAddr++);
                    --len;
                    ++addr;
                    ++i;

                    // Update the displayed variable
                    ++bytesDownloaded;
                }
            }

            // Download is complete if the EOF tag is found
            // Break again to get out of the for(;;) loop
            if (i == 0 && addr == ADDR_EOF) break;
        }
        
        switch (memoryType) {
          case EXTERNAL_SRAM:
          case INTERNAL_SRAM:
            _CKERR_VOID(
              jtagWriteMemory(addr+offset, len, gBytes, MEM_MODE_LONG_SHORT) );
            break;
          case FLASH_MEMORY:
            _CKERR_VOID( _progWriteMemory(addr, len, gBytes) );
            break;
          case SEEPROM_MEMORY:
            _CKERR_VOID(
                _progWriteMemory(alignedAddr, SEEPROM_BLOCK_SIZE, gBytes) );
            break;
        }

        dispDecimal((data_8) ((bytesDownloaded / 1000) % 100));
    }
    
    // Reset the CPSU completely and run
    _CKERR_VOID( jtagResetCpsu() );
    _CKERR_VOID( jtagResetAndRun() );
}


/*=========================================================================
| STATIC FUNCTIONS (PROGRAMMED MEMORY)
 ========================================================================*/
void _progWriteMemory (data_32 addr, data_16 len, data_8 bytes[])
{
    data_8  lBytes[6];

    // Assemble the address and length information in a single
    // buffer to speed things up.  PROG_ADDR AND PROG_COUNT
    // are right next to each other so we can do all 6 bytes at
    // once.

    lBytes[0] = (addr >> 24) & 0xff;
    lBytes[1] = (addr >> 16) & 0xff;
    lBytes[2] = (addr >>  8) & 0xff;
    lBytes[3] = (addr      ) & 0xff;
    
    lBytes[4] = (len >> 8) & 0xff;
    lBytes[5] = (len     ) & 0xff;
    
    // Write the address and length info to the target
    _CKERR_VOID(
        jtagWriteMemory(PROG_ADDR,   6, lBytes, MEM_MODE_LONG_SHORT) );
    
    // Write the data to the target
    _CKERR_VOID(
        jtagWriteMemory(PROG_DATA, len, bytes, MEM_MODE_LONG_SHORT) );
    
    // Issue the program command
    _CKERR_VOID( _issueProgCommand(PROG_CMD_PROGRAM) );
}


static void _targetWriteByte (data_32 addr, data_8 value)
{
    data_8 outData[1];
    outData[0] = value;

    _CKERR_VOID( jtagWriteMemory(addr, 1, outData, MEM_MODE_LONG_SHORT) );
}

static void _issueProgCommand (data_8 command)
{
    data_8 lBytes[2];

    // Write the command and control in a single operation
    // since these bytes are back to back in the target address space
    lBytes[0] = command;
    lBytes[1] = PROG_CTL_GO;
    
    _CKERR_VOID
        ( jtagWriteMemory(PROG_COMMAND, 2, lBytes, MEM_MODE_LONG_SHORT) );

    // Wait for the target to receive the command
    do {
        _CKERR_VOID
            ( jtagReadMemory(PROG_CONTROL, 1, lBytes, MEM_MODE_LONG_SHORT) );
    } while (lBytes[0] == PROG_CTL_GO);

    // Wait for the target to finish processing the command
    do {
        _CKERR_VOID
            ( jtagReadMemory(PROG_STATUS, 1, lBytes, MEM_MODE_LONG_SHORT) );
    } while (lBytes[0] == command);

    // If this is an error, return the error code
    switch (lBytes[0] & 0xff) {
      case PROG_ERR_BAD_COMMAND:
      case PROG_ERR_UNKNOWN_DEVICE:
      case PROG_ERR_MEMORY_WRITE:
      case PROG_ERR_MEMORY_ERASE:
      case PROG_ERR_VERIFY:
        gDlError = lBytes[0];
        break;
    }
}

static void _dlProgrammer (MemoryType memoryType, data_8 algorithmName[])
{
    DataRecord *programmerCode;
    int         i = 0;

    // Set the display to "Programmer Download"
    dispHex(0xfd);

    switch (memoryType) {
      case FLASH_MEMORY:
        programmerCode = FLASH_PROGRAMMER_CODE; break;
      case SEEPROM_MEMORY:
        programmerCode = SEEPROM_PROGRAMMER_CODE; break;
      default:
        // This should never happen
        return;
    }
    
    // Download the flash programming algorithm to the target board
    while (programmerCode[i].address != ADDR_EOF) {
        _CKERR_VOID
            ( jtagWriteMemory(programmerCode[i].address +INTERNAL_RAM_OFFSET,
                              programmerCode[i].numBytes,
                              programmerCode[i].bytes,
                              MEM_MODE_LONG_SHORT) );
        ++i;
    }

    // Program the code mappers and start the algorithm
    // 
    // Set the code mappers on the target to internal ram so
    // that the programming algorithm may be started.
    // This is common for all the programming algorithms.
    _CKERR_VOID( _targetWriteByte(REG_ENAREG, 0x01) );

    // Disable CMAP1
    _CKERR_VOID( _targetWriteByte(REG_CMAP1_CTL, 0x00) );    

    // Set TAR to 0x00010000
    _CKERR_VOID( _targetWriteByte(REG_CMAP2_TAR_0, 0x00) );
    _CKERR_VOID( _targetWriteByte(REG_CMAP2_TAR_1, 0x01) );
    _CKERR_VOID( _targetWriteByte(REG_CMAP2_TAR_2, 0x00) );

    // Set CMAP2 to 0 in CODE
    _CKERR_VOID( _targetWriteByte(REG_CMAP2_SRC, 0x00) );

    // Set CMAP2 to 4K
    _CKERR_VOID( _targetWriteByte(REG_CMAP2_CTL, 0x2c) );
    
    if (memoryType == FLASH_MEMORY) {
        _CKERR_VOID( _targetWriteByte(REG_MIUMODE, 0x01) );
    }

    else if (memoryType == SEEPROM_MEMORY) {
        // Set CRU visibility
        _CKERR_VOID( _targetWriteByte(REG_DMAP3_CTL,   0x01) );

        // Set TAR to 0x00010000
        _CKERR_VOID( _targetWriteByte(REG_DMAP5_TAR_0, 0x00) ); 
        _CKERR_VOID( _targetWriteByte(REG_DMAP5_TAR_1, 0x01) );
        _CKERR_VOID( _targetWriteByte(REG_DMAP5_TAR_2, 0x00) );
        
        // Set DMAP2 to 0 in CODE        
        _CKERR_VOID( _targetWriteByte(REG_DMAP5_SRC,   0x00) ); 
        _CKERR_VOID( _targetWriteByte(REG_DMAP5_CTL,   0x2d) );

        // Set MIU to serial write
        _CKERR_VOID( _targetWriteByte(REG_MIUMODE,     0x08) );
        
        // Reset serial control        
        _CKERR_VOID( _targetWriteByte(REG_MIUSCTL,     0xe7) );
    }
    
    _CKERR_VOID( jtagResetAndRun() );
    
    // Set the display to "Flash Algorithm"
    dispHex(0xfa);
    
    // Send over the programming algorithm name including the '\0'
    _CKERR_VOID( jtagWriteMemory(PROG_DEVICE, strlen(algorithmName)+1,
                                 algorithmName, MEM_MODE_LONG_SHORT) );
    _CKERR_VOID( _issueProgCommand(PROG_CMD_SET_DEVICE) );
    

    if (memoryType == FLASH_MEMORY) {
        // Erase the flash; set the display to "Flash Erase"
        dispHex(0xfe);
        _CKERR_VOID( _issueProgCommand(PROG_CMD_BULK_ERASE) );
    }
    
    // Signal that this phase is done
    dispHex(0xf0);
}


/*=========================================================================
| STATIC FUNCTIONS (Accessing target data from own memory)
 ========================================================================*/
static void memMap (data_32 mem_addr)
{
    static bit first = 1;

    /* If the address is already included in the block, return.
       If this is the first time, force a mapping.
    */
    if (first                            == 0          &&
        gMemOffset                       <= mem_addr   &&
        gMemOffset +DATA_BUFFER_SIZE -1  >= mem_addr)
        return;

    first = 0;


    /* Map in the External Memory, which sits at 32-bit address 0x800000 */
    mem_addr   &= DATA_BUFFER_MASK;
    gMemOffset  = mem_addr;
    mem_addr   += EXT_MEM_BASE;

    /* Disable it first */
    DMAP5_CTL   = 0x00;

    /* Map the External Memory address */
    DMAP5_TAR_2 = (mem_addr >> 24) & 0xff;
    DMAP5_TAR_1 = (mem_addr >> 16) & 0xff;
    DMAP5_TAR_0 = (mem_addr >>  8) & 0xff;

    /* Map it to 0x8000 in the XDATA space */
    DMAP5_SRC   = 0x80;

    /* Enable it and set the size to that of the data buffer */
    DMAP5_CTL   = 0x20 | DATA_BUFFER_EXP;
}


static data_8 memRead (data_32 addr)
{
    memMap(addr);
    return gExtMem[addr - gMemOffset];
}

static void memWrite (data_32 addr, data_8 value)
{
    memMap(addr);
    gExtMem[addr - gMemOffset] = value;
}


/*=========================================================================
| STATIC FUNCTION
 ========================================================================*/
static void setupMIU (data_32 capacityKB)
{
    int index = 0;
    int piosel;

    gBytes[0] = 0x01;
    _CKERR_VOID(
        jtagWriteMemory(REG_MIUMODE, 1, gBytes, MEM_MODE_LONG_SHORT) );
    
    // Here's a table of memory size versus piosel
    //
    //   Size      Index   ==>    PIOSEL (16 bits)
    //   256 kB      0     ==>    00000000 00000001
    //   512 kB      1     ==>    00000000 00000101
    //  1024 kB      2     ==>    00000000 00001101
    //  2048 kB      3     ==>    00000000 00011101
    //    ...
    //
    // The lowest order bit should be 1 to signal that
    // the memory device is parallel.
    capacityKB >>= 9;
    while (capacityKB != 0) { ++index; capacityKB >>= 1; }
    
    piosel = ~(0xfffc << index) & ~0x02;
    
    gBytes[0] = (piosel     ) & 0xff;
    gBytes[1] = (piosel >> 8) & 0xff;

    _CKERR_VOID(
        jtagWriteMemory(REG_PIOSEL_0, 2, gBytes, MEM_MODE_LONG_SHORT) );
}
