/*=========================================================================
| (c) 2000  Triscend Corporation
|--------------------------------------------------------------------------
| Project : Embedded E5 Jtag Library
|
| File    : te5jtag.c
 ========================================================================*/

/*=========================================================================
| INCLUDES
 ========================================================================*/
#include "te5jtag.h"
#include "te5utils.h"


/*=========================================================================
| PROTOTYPES
 ========================================================================*/
static void _readBusStat (data_8 dataOut[]);
static data_8 jtagShiftIR_checkConn (data_8 opcode);

    
/*=========================================================================
| MACROS
 ========================================================================*/
// 1) Assign var to the result of function()
// 2) On error, return
#define _CKERR(var, function) \
    { \
        jtagError = JTAG_OK; \
        var = function; \
        if (jtagError != JTAG_OK) return; \
    }

// 1) Assign var to the result of function()
// 2) On error, return retval
#define _CKERR_RET(var, function, retval) \
    { \
        jtagError = JTAG_OK; \
        var = function; \
        if (jtagError != JTAG_OK) return retval; \
    }

// 1) Call function()
// 2) On error, return
#define _CKERR_VOID(function) \
    { \
        jtagError = JTAG_OK; \
        function; \
        if (jtagError != JTAG_OK) return; \
    }
        
// 1) Call function()
// 2) On error, return retval
#define _CKERR_VOID_RET(function, retval) \
    { \
        jtagError = JTAG_OK; \
        function; \
        if (jtagError != JTAG_OK) return retval; \
    }


// MCU status macros
#define _mcuFrozen(status)   (((status)[0] & BUSTAT_MCU_FROZEN) != 0)
#define _mcuRunning(status)  (((status)[0] & 0x0f) == BUSTAT_MCU_RUNNING)
#define _mcuInReset(status)  (((status)[0] & BUSTAT_MCU_RESET) != 0)


/*=========================================================================
| GLOBAL VARIABLES
 ========================================================================*/
data_8 jtagError;


/*=========================================================================
| STATIC VARIABLES
 ========================================================================*/
static data_8 _jtagControl[2];


/*=========================================================================
| CONSTRUCTOR
 ========================================================================*/
void jtagInitialize ()
{
    _jtagControl[0] = 0x00;
    _jtagControl[1] = 0x80;
    jtagError       = JTAG_OK;
    
    tapInitialize();
    
    _CKERR_VOID(jtagAreYouThere());
}


/*=========================================================================
| DESTRUCTOR
 ========================================================================*/
void jtagDeinitialize ()
{
    tapDeinitialize();
}


/*=========================================================================
| BASIC BUS STATUS, CONTROL, LINK, AND ID ROUTINES
 ========================================================================*/
data_16 jtagBusStatus ()
{
    data_8  dataOut[2];
    data_16 busStatus;

    _CKERR_VOID_RET(_readBusStat(dataOut), 0);

    busStatus  = (dataOut[0]);
    busStatus |= (dataOut[1] << 8);

    return busStatus;
}

void jtagBusControl (data_16 buscon)
{
    data_8 dataOut[2];

    // First check the connection
    _CKERR_VOID(jtagShiftIR_checkConn(BUSTAT));

    // Assemble the control bytes and shift them in
    _jtagControl[0]  = ((buscon & 0xff));
    _jtagControl[1] |= ((buscon >> 8) & 0x7f);

    _CKERR_VOID
        (tapJtagScan(JTAG_REG_DR, 11,
                     JTAG_STATE_RUN_TEST_IDLE, _jtagControl, dataOut));
}

int jtagAreYouThere ()
{
    // Reset the TAP controller
    _CKERR_VOID_RET(tapJtagReset(), 0);

    // Check for the presence of the Triscend IDCODE
    _CKERR_VOID_RET(jtagShiftIR_checkConn(IDCODE), 0);

    return 1;
}

data_32 jtagDeviceID ()
{
    data_8 dataIn[4];
    data_8 dataOut[4];

    _CKERR_VOID_RET(jtagAreYouThere(), 0);
    
    _CKERR_VOID_RET(tapJtagScan(JTAG_REG_DR, 32,
                                  JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut),
                      0);

    return jtagToData32(dataOut);
}


/*=========================================================================
| EXECUTION ROUTINES:
|   Reset Cpsu
|   Reset and Run
|   Reset and Halt
|   Run
|   Stop
 ========================================================================*/
void jtagResetCpsu ()
{
    _CKERR_VOID( jtagBusControl(0x08) );
    _CKERR_VOID( jtagBusControl(0x02) );
}

void jtagResetAndRun ()
{
    // Check the bus status
    data_8 dataOut[2];
    _CKERR_VOID(_readBusStat(dataOut));

    // bclk_status, brst_, mcu_frozen_, sw_bpevt, mcu_inreset
    if (_mcuInReset(dataOut)) {
        // MCU is already reset check if it is resetted by JTAG
        if ((_jtagControl[0] & (BUSCON_MCU_RESET | BUSCON_FORCE_BRST)) == 0)
            // _jtagControl[0] = BUSCON_FORCE_NOBRST;
            _jtagControl[0] = 0;
    } else {
        // MCU needs to be reset first
        _jtagControl[0] |= BUSCON_MCU_RESET;
        _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 11,
                                  JTAG_STATE_RUN_TEST_IDLE,
                                  _jtagControl, dataOut));
    }
    
//// Clear all except the FORCE_NOBRST bit and start running
    // _jtagControl[0] &= BUSCON_FORCE_NOBRST;  

// Clear buscon and start running
    _jtagControl[0] = 0;  
    _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 11,
                            JTAG_STATE_RUN_TEST_IDLE,
                            _jtagControl, dataOut));
}

void jtagResetAndHalt ()
{
    // Check the bus status
    data_8 dataOut[2];
    _CKERR_VOID(_readBusStat(dataOut));

    // Reset the CSOC
    _jtagControl[0] |= BUSCON_MCU_RESET;
    _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 11,
                              JTAG_STATE_RUN_TEST_IDLE,
                              _jtagControl, dataOut));

    // Check to see if we actually are in reset
    _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 11,
                            JTAG_STATE_RUN_TEST_IDLE,
                            _jtagControl, dataOut));

    if (!_mcuInReset(dataOut)) {
        jtagError = JTAG_ERR_CANT_STOP_TARGET;
    }
}

void jtagRun ()
{
    // Check the bus status
    data_8 dataOut[2];
    _CKERR_VOID(_readBusStat(dataOut));

    // If the MCU is currently running, ignore this command
    if (_mcuRunning(dataOut))
        return;

    // _jtagControl[0] &= BUSCON_FORCE_NOBRST;
    _jtagControl[0] = 0;
    _CKERR_VOID(jtagShiftIR_checkConn(BUSTAT));
    
    _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 11, JTAG_STATE_RUN_TEST_IDLE,
                              _jtagControl, dataOut));
}

void jtagStop ()
{
    // Check the bus status
    data_8 dataOut[2];
    _CKERR_VOID(_readBusStat(dataOut));

    // Freeze the MCU
    _jtagControl[0] |= BUSCON_MCU_FREEZE;
    _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 11,
                            JTAG_STATE_RUN_TEST_IDLE,
                            _jtagControl, dataOut));

    // Check to make sure it is stopped now
    _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 11,
                            JTAG_STATE_RUN_TEST_IDLE,
                            _jtagControl, dataOut));
    
    if (!_mcuInReset(dataOut)) {
        jtagError = JTAG_ERR_CANT_STOP_TARGET;
    }
}


/*=========================================================================
| PUBLIC METHODS
 ========================================================================*/
void jtagReadMemory (data_32          addr,
                     data_32          count,
                     data_8           bytes[],
                     JtagMemMode      mode)
{
    data_8   dataIn[5];
    data_8   dataOut[5];
    data_32  currByte;

    switch (mode) {
      case MEM_MODE_DEFAULT:         
      case MEM_MODE_LONG_SHORT:
        _CKERR_VOID( jtagShiftIR_checkConn(MREAD_LONG) );
        
        data32ToJtag(dataIn, addr);
        ++addr;
        dataIn[4] = 1;               // Set bus request bit
        _CKERR_VOID(tapJtagScan
            (JTAG_REG_DR, 33, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
        
        if ( count > 1 ) {
            data_16 shortAddr = addr & 0xffff;
            if (shortAddr)           // Use MREAD instead of MREAD_LONG
                _CKERR_VOID( jtagShiftIR_checkConn(MREAD) );
            
            for (currByte = 0; currByte < count-1; currByte++) {
                if (!shortAddr) {    // Use MREAD_LONG
                    if (currByte != 0) {
                        _CKERR_VOID( jtagShiftIR_checkConn(MREAD_LONG) );
                    }
                    
                    data32ToJtag(dataIn, addr);
                    _CKERR_VOID(tapJtagScan
                        (JTAG_REG_DR, 33, JTAG_STATE_RUN_TEST_IDLE,
                         dataIn, dataOut));
                    // Set IR back to MREAD for next loop
                    _CKERR_VOID( jtagShiftIR_checkConn(MREAD) );
                } else {             // Use MREAD
                    dataIn[2] = 1;   // Set bus request bit
                    data16ToJtag(dataIn, shortAddr);
                    _CKERR_VOID(tapJtagScan
                        (JTAG_REG_DR, 17, JTAG_STATE_RUN_TEST_IDLE,
                         dataIn, dataOut));
                }
                ++addr;
                ++shortAddr;
                bytes[currByte] = ((dataOut[0] >> 1) | (dataOut[1] << 7));
            }
        }

        // Get the last byte
        _CKERR_VOID( jtagShiftIR_checkConn(MVERIFY) );
        dataIn[0] = 0;    // Clear bus request bit
        _CKERR_VOID(tapJtagScan
            (JTAG_REG_DR, 9, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
        bytes[count-1] = (dataOut[0] >> 1) | (dataOut[1] << 7);
        break;
        
      case MEM_MODE_LONG:    // = 1 in Wiggler's scheme
        _CKERR_VOID( jtagShiftIR_checkConn(MREAD_LONG) );

        // Shift in the address
        data32ToJtag(dataIn, addr);
        dataIn[4] = 1;    // Set bus request bit
        _CKERR_VOID(tapJtagScan
            (JTAG_REG_DR, 33, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
        ++addr;

        for (currByte = 0; currByte < count - 1; currByte++) {
            // Shift the next n-1 addresses and get the first n-1 value
            data32ToJtag(dataIn, addr);
            _CKERR_VOID(tapJtagScan
                (JTAG_REG_DR, 33, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
            bytes[currByte] = ((dataOut[0] >> 1) | (dataOut[1] << 7));
            ++addr;
        }
        dataIn[4] = 0;    // Clear bus request bit

        // Get the last value
        _CKERR_VOID(tapJtagScan
            (JTAG_REG_DR, 33, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
        bytes[count-1] = ((dataOut[0] >> 1) | (dataOut[1] << 7));
        break;
        
      case MEM_MODE_SHORT:    // = 3 in Wiggler's scheme
        _CKERR_VOID( jtagShiftIR_checkConn(MREAD) );
        
        data16ToJtag(dataIn, addr);
        dataIn[2] = 1;        // Set bus request bit
        _CKERR_VOID(tapJtagScan
            (JTAG_REG_DR, 17, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
        for (currByte = 0; currByte < count - 1; currByte++) {
            data16ToJtag(dataIn, addr);
            _CKERR_VOID(tapJtagScan
                (JTAG_REG_DR, 17, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
            bytes[currByte] = ((dataOut[0] >> 1) | (dataOut[1] << 7));
            ++addr;
        }

        // Get the last byte
        _CKERR_VOID( jtagShiftIR_checkConn(MVERIFY) );
        dataIn[0] = 0;        // Clear bus request bit
        _CKERR_VOID(tapJtagScan
            (JTAG_REG_DR, 9, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
        bytes[count-1]= (dataOut[0] >> 1) | (dataOut[1] << 7);
        break;

      case MEM_MODE_VERIFY:
        if (count != 1) {
            jtagError = JTAG_ERR_MEM_INVALID_MODE;
            return;
        }

        _CKERR_VOID( jtagShiftIR_checkConn(MVERIFY) );

        dataIn[0] = 0;    // Clear bus request bit
        dataOut[0] = 0;
        // Just reuse currByte as a index in this loop
        for (currByte = 0; currByte < 100; currByte++) {
            _CKERR_VOID(tapJtagScan
                (JTAG_REG_DR, 9, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
            if (dataOut[0] & 0x01)
                break;
        }

        if (!(dataOut[0] & 0x01)) {
            jtagError = JTAG_ERR_MEM_VERIFY_MISMATCH;
        }

        bytes[0] = ((dataOut[0] >> 1) | (dataOut[1] << 7));
        break;

      default:
        jtagError = JTAG_ERR_MEM_INVALID_MODE;
        break;
    }
}

void jtagWriteMemory (data_32       addr,
                      data_32       count,
                      data_8        bytes[],
                      JtagMemMode   mode)
{
    data_8   dataIn[6];
    data_8   dataOut[6];
    data_16  shortAddr;
    data_32  currByte;

    switch (mode) {
      case MEM_MODE_DEFAULT:
      case MEM_MODE_LONG_SHORT:
        _CKERR_VOID( jtagShiftIR_checkConn(MWRITE_LONG) );

        data32ToJtag(dataIn, addr);
        ++addr;

        dataIn[4] = bytes[0];
        dataIn[5] = 1;    // Set bus request bit
        // Get the first long transfer
        _CKERR_VOID(tapJtagScan
            (JTAG_REG_DR, 41, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
        shortAddr = addr & 0xffff;
        if (shortAddr) {
            // Use MWRITE instead of MWRITE_LONG
            _CKERR_VOID( jtagShiftIR_checkConn(MWRITE) );
        }

        for (currByte = 1; currByte < count; currByte++) {
            if (!shortAddr) {    // Use MWRITE_LONG
                if (currByte != 1) {
                    _CKERR_VOID( jtagShiftIR_checkConn(MWRITE_LONG) );
                }

                data32ToJtag(dataIn, addr);
                dataIn[4] = bytes[currByte];
                _CKERR_VOID(tapJtagScan
                    (JTAG_REG_DR, 41, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));

                // Set it back to MWRITE for next loop
                _CKERR_VOID( jtagShiftIR_checkConn(MWRITE) );
            } else {              // Use MWRITE
                dataIn[3] = 1;    // Set bus request bit
                dataIn[2] = bytes[currByte];
                data16ToJtag(dataIn, shortAddr);
                _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 25,
                    JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
            }
            ++addr;
            ++shortAddr;
        }
        break;
      
      case MEM_MODE_LONG:     // = 1 in Wiggler's scheme
        _CKERR_VOID( jtagShiftIR_checkConn(MWRITE_LONG) );

        dataIn[5] = 1;        // Set bus request bit
        for (currByte = 0; currByte < count; currByte++) {
            data32ToJtag(dataIn, addr);
            dataIn[4] = bytes[currByte];
            _CKERR_VOID(tapJtagScan
                (JTAG_REG_DR, 41, JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
            ++addr;
        }
        break;
        
      case MEM_MODE_SHORT:    //  = 3 in Wiggler's scheme
        _CKERR_VOID( jtagShiftIR_checkConn(MWRITE) );
        
        dataIn[3] = 1;        // Set bus request bit 
        for (currByte = 0; currByte < count; currByte++) {
            data16ToJtag(dataIn, addr);
            dataIn[2] = bytes[currByte];
            _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 25,
                JTAG_STATE_RUN_TEST_IDLE, dataIn, dataOut));
            ++addr;
        }
        break;
        
      default:
        jtagError = JTAG_ERR_MEM_INVALID_MODE;
        break;
    }
}


/*=========================================================================
| PRIVATE METHODS
 ========================================================================*/
static void _readBusStat (data_8 dataOut[])
{
    _CKERR_VOID(jtagShiftIR_checkConn(BUSTAT));
    
    /* If this is the first time any BUSTAT command has been
       asserted, read back the status non-destructively.

       This is achieved by shifting in 22 bits, the last 11 of
       which are the status that was read back while shifting the
       first 11.

       1) Shift in 11 bogus bits, getting the status in the process.
       2) Set a valid control based on the status, and shift in 11 bits.

       Only the last 11 bits are used for the JTAG update, ensuring
       that the target execution state is not changed by this process.
    */

    if (_jtagControl[1] == 0x80) {
        data_8 bogus[2];

        _jtagControl[0] = 0;
        _jtagControl[1] = 0;
        
        _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 11,
                                  JTAG_STATE_SHIFT_DR, _jtagControl,
                                  dataOut));

        if (_mcuFrozen(dataOut))
            _jtagControl[0] |= BUSCON_MCU_FREEZE;

        _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 11,
                                  JTAG_STATE_RUN_TEST_IDLE, _jtagControl,
                                  bogus));
    }
    else
        _CKERR_VOID(tapJtagScan(JTAG_REG_DR, 11,
                                  JTAG_STATE_RUN_TEST_IDLE, _jtagControl,
                                  dataOut));
}


/*=========================================================================
| PRIVATE METHODS
 ========================================================================*/
static data_8 jtagShiftIR (data_8 opcode)
{
    data_8 in[1];
    data_8 out[1];
    
    in[0] = opcode;
    tapJtagScan(JTAG_REG_IR, 4, JTAG_STATE_RUN_TEST_IDLE, in, out);
    return out[0];
}


/*=========================================================================
| PRIVATE METHODS
 ========================================================================*/
static data_8 jtagShiftIR_checkConn (data_8 opcode)
{
    data_8 dataOut = jtagShiftIR(opcode);

    if ( (dataOut & 0x7) != 0x01 ) {
        if ( (dataOut & 0x7) == 0x05 ) {
            jtagError = JTAG_ERR_TARGET_SECURITY_ERROR;
        }
        else {
            jtagError = JTAG_ERR_TARGET_COMM_FAILURE;
        }
    }

    return dataOut;
}
