/*=========================================================================
| (c) 2000  Triscend Corporation
|--------------------------------------------------------------------------
| Project : Embedded E5 Jtag Library
|
| File    : te5tap.c
 ========================================================================*/

/*=========================================================================
| INCLUDES
 ========================================================================*/
#include "te5tap.h"
#include "te5utils.h"

#define PROTOTYPE_ONLY
#include "JtagTap.h"


/*=========================================================================
| STATIC VARIABLES
 ========================================================================*/
static JtagState _tapState;


/*=========================================================================
| STATIC PROTOTYPES
 ========================================================================*/
static void _gotoState (JtagState state);


/*=========================================================================
| MACROS
 ========================================================================*/
/* This should be customized for your specific JTAG TAP hardware */
#define TCK_BIT 0
#define TMS_BIT 1
#define TDI_BIT 2
#define TDO_BIT 3

/* patterns */
#define TCK0  (0)
#define TMS0  (0)
#define TDI0  (0)
#define TDO0  (0)

#define TCK1  (1<<TCK_BIT)
#define TMS1  (1<<TMS_BIT)
#define TDI1  (1<<TDI_BIT)
#define TDO1  (1<<TDO_BIT)

/* interface macros */
#define _tapWrite(pattern)  (regJtagTap = (pattern))
#define _tapRead()          ((regJtagTap & TDO1) != 0)
#define _tapShift(tdi_tms) \
    (_tapWrite((tdi_tms) | TCK0), _tapWrite((tdi_tms) | TCK1), _tapRead())

// Prototype
void tapJtagReset ();
    
/*=========================================================================
| CONSTRUCTOR
 ========================================================================*/
void tapInitialize ()
{
    _tapState = JTAG_STATE_TEST_LOGIC_RESET;
    tapJtagReset();
}


/*=========================================================================
| DESTRUCTOR
 ========================================================================*/
void tapDeinitialize ()
{
    // Do nothing
}


/*=========================================================================
| FUNCTIONS
 ========================================================================*/
void tapJtagReset ()
{
    // Set the initial values to 1 and perform some
    // dummy operations to initialize the TAP controller
    _tapWrite(TDO1 | TMS1 | TCK1);
    _tapShift(TDI0 | TMS1);
    _tapShift(TDI1 | TMS1);
    _tapShift(TDI0 | TMS1);

    // Write five 1s to TMS to force a transition to the
    // TEST_LOGIC_RESET state, then issue a 0 to put the
    // JTAG state machine into RUN_TEST_IDLE
    _tapShift(TDI0 | TMS1);
    _tapShift(TDI0 | TMS1);
    _tapShift(TDI0 | TMS1);
    _tapShift(TDI0 | TMS1);
    _tapShift(TDI0 | TMS1);
    _tapShift(TDI0 | TMS0);
    
    _tapState = JTAG_STATE_RUN_TEST_IDLE;
}


void tapJtagScan (JtagShiftReg  reg,
                  data_8        bits,
                  JtagState     finalState,
                  data_8        dataIn[],
                  data_8        dataOut[])
{
    data_8 i;
    data_8 tdo;

    // First go to the appropriate shifting state
    _gotoState((reg == JTAG_REG_IR) ?
               JTAG_STATE_SHIFT_IR : JTAG_STATE_SHIFT_DR);

    // Shift all but the final bit, LSB first
    for (i = 0; i < bits -1; ++i) {
        tdo = _tapShift(((getBit(dataIn, i)) ? TDI1 : TDI0) | TMS0);
        setBit(dataOut, i, tdo);
    }
    
    // If the final state requested is the current shift state,
    // do not set TMS1 for the final bit.
    if (finalState == _tapState) {
        // Shift the final bit, setting TMS to 0
        tdo = _tapShift(((getBit(dataIn, i)) ? TDI1 : TDI0) | TMS0);
        setBit(dataOut, i, tdo);
    }
    else {
        // Shift the final bit, setting TMS to 1
        tdo = _tapShift(((getBit(dataIn, i)) ? TDI1 : TDI0) | TMS1);
        setBit(dataOut, i, tdo);
    
        // Set the current state to EXIT1_IR or EXIT1_DR
        _tapState = (reg == JTAG_REG_IR) ?
                        JTAG_STATE_EXIT1_IR : JTAG_STATE_EXIT1_DR;
    
        // Goto the final state requested.
        _gotoState(finalState);
    }
}


/*=========================================================================
| PRIVATE FUNCTION
 ========================================================================*/
static void
_gotoState(JtagState state)
{
    // If the requested state is the same, just return
    if (_tapState == state)  return;

    // Currently, this function only supports:
    //
    //   RUN_TEST_IDLE ==> SHIFT_IR
    //   RUN_TEST_IDLE ==> SHIFT_DR
    //   EXIT1_IR      ==> RUN_TEST_IDLE
    //   EXIT1_DR      ==> RUN_TEST_IDLE
    //
    switch (_tapState) {
      case JTAG_STATE_RUN_TEST_IDLE:
        switch (state) {
          case JTAG_STATE_SHIFT_IR:
            _tapShift(TMS1);
            _tapShift(TMS1);
            _tapShift(TMS0);
            _tapShift(TMS0);
            _tapState = JTAG_STATE_SHIFT_IR;
            break;
            
          case JTAG_STATE_SHIFT_DR:
            _tapShift(TMS1);
            _tapShift(TMS0);
            _tapShift(TMS0);
            _tapState = JTAG_STATE_SHIFT_DR;
            break;
        }
        break;
        
      case JTAG_STATE_EXIT1_IR:
      case JTAG_STATE_EXIT1_DR:
        if (state == JTAG_STATE_RUN_TEST_IDLE) {
            _tapShift(TMS1);
            _tapShift(TMS0);
            _tapState = JTAG_STATE_RUN_TEST_IDLE;
        }
        break;
    }
}
