/*------------------------------------------------------------------------------

  datagram.c

  (c) Copyright 1999 by Triscend Corporation (www.triscend.com).
  All Rights Reserved.

  This code creates a reliable serial communications channel to transfer
  large files between a PC and the Triscend evaluation board.

  2/10/99.
------------------------------------------------------------------------------*/
 
#include <stdio.h>


#include "datagram.h"
#include "serial.h"
#include "decrypt.h"
#include "download.h"

 
//--------------------------------------------------------------------
//  Physical Layer Code is in:   serial.c
//------------------------- END PHYSICAL LAYER CODE ------------------

#define SLIP_ESC 0xdb
#define SLIP_END 0xc0
#define SLIP_ESC_SUB 0xdd
#define SLIP_END_SUB 0xdc

// Datagram = (Length, CRC, up to 256 bytes of data)
// Allow a 2 byte field for datagram length for flexibility  (currently
// are just exceeding one byte limit at 258 entries).
#define DATAGRAM_LENGTH 0
#define DATAGRAM_CHECKSUM 2
#define DATAGRAM_DATA 3
#define DATAGRAM_SIZE 258

// ASCII control characters - used to communicate status back to PC.
// Acknowledge packet
#define ACK 0x06
// Negative Acknowledge Packet - packet was corrupt e.g. bad checksum
#define NAK 0x15
// Negative acknowledge Download - will not be activated. e.g. bad MAC
#define CAN 0x18
// Acknowledge Download - will be installed
#define ETX 0x30

// Buffer for datagram data received
static unsigned char datagram[DATAGRAM_SIZE];
// Index of next char to be written into datagram[]
static unsigned short datagram_index;
// Total number of chars received up to now
static long chars_received; 
// Total number of datagrams received
static unsigned char datagrams_received=0;

// Flag set when entire file is downloaded
static unsigned char communication_finished=0;


static unsigned char checksum_ok() {
	short i;
	unsigned char checksum;
	
   checksum=0;
	for (i=DATAGRAM_DATA;i<datagram_index;i++) {
		checksum+=datagram[i];
   }

   return (checksum==datagram[DATAGRAM_CHECKSUM]);
} // checksum_ok

static void process_datagram(void);

static void slip_datagram_end()
{
      
	// Check completed datagram.
   if ((datagram_index!=0)&&    // Must have some data in it!
                                // Must be the right length
       (datagram_index==((datagram[DATAGRAM_LENGTH]<<8) + datagram[DATAGRAM_LENGTH+1])) &&
                                // Checksum must be as expected
       (checksum_ok())) {
			// The packet is OK - it can be decrypted and written to Flash
           process_datagram();
           if (!communication_finished) {
         	  // Tell the PC its OK so it will send next packet
       	  putchar(ACK);
          	} else {
             // Acknowledge complete download
             putchar(ETX);
           }
	} else putchar(NAK);
	
	// Initialise for next datagram.
	datagram_index=0;
} // slip_datagram_end


/* This routine strips out the SLIP substitutions to output the exact same sequence
   of bytes as was passed to slip_transmit() on the PC. SLIP substitutions are necessary
   since any binary value is legal in the data and we need some reserved characters
   for datagram framing. */
static void receive_slip(c)
unsigned char c;
{
	static char next_char_escaped=0;
	
   chars_received++;
	if (c==SLIP_ESC) {
		next_char_escaped=1;
	} else if (next_char_escaped) {
		next_char_escaped=0;
		if (c==SLIP_ESC_SUB) {
			c=SLIP_ESC;
		} else if (c==SLIP_END_SUB) {
			c=SLIP_END;
		} else {
			printf("Error, Bad SLIP code substitution %c\n",c);
			// exit(1);
		}
		datagram[datagram_index]=c;
 		datagram_index++;
	} else if (c==SLIP_END) {  
		// N.B. need to check char is not escaped AND it is SLIP_END
		slip_datagram_end();
	} else {
		datagram[datagram_index]=c;
 		datagram_index++;
	}
}  // receive_slip


//-------------------------- END OF SLIP FRAMING CODE --------------------


               
// Called after receiving a valid datagram, after CRC check.
static void process_datagram() {

	unsigned short cindex;

   datagrams_received++;
	for (cindex=DATAGRAM_DATA;cindex<datagram_index;cindex++) {
		receive(datagram[cindex]);  // pass the data on to the application level code
 	}
 } // process_datagram
 
 

//-------------------------- END DATAGRAM CODE ---------------------------

//--------------- Interface with application level code. ------------------

void close_channel (void) {
   communication_finished=1;
} // close_channel

void reset_channel (void) {
   putchar(CAN);  // Inform PC entire download has failed.
   initialise_decrypt();
   chars_received=0;
	datagram_index=0;
	communication_finished=0;
	
} // reset_channel

void open_channel (void) {

   unsigned char c;
   
	serial_init();
	initialise_decrypt();
	chars_received=0;
	datagram_index=0;
	communication_finished=0;
	
   // This is where the actual communication happens.
   while (!communication_finished) {
   	c=_getkey();     // Get a character from physical layer
   	receive_slip(c); // process it
   	                 // This routine calls application level code. The application code
                     	 // will call close_channel() when its communication is finished.
   }
} /* open channel */


