/*------------------------------------------------------------------------------

install.c

(c) Copyright 1999 by Triscend Corporation (www.triscend.com).  All Rights Reserved.

This code saves the downloaded image into Flash and after a succesful download,
switches to the new configuration and program.


------------------------------------------------------------------------------*/


#include "download.h"
#include "install.h"
#include "nvdata.h"

// Offsets of segments in AMD 29LV0002BB chip installed on Triscend Development Board.
// Taken from Table 3. in AM29LV002B data sheet.
code const unsigned long flash_segment_offsets[7]={
	0L,   // Segment 0 holds constant (non-downloaded) bootsrap code
   0x4000L,0x6000L,0x8000L,0x10000L, // Half 0, 112K for downloaded code, in 4 segments
   0x20000L,0x30000L                 // Half 1, 128K for downloaded code, in 2, 64K segments
   };

// If the CSoC booted from Half 0, then it will download into Half 1 and vice versa.
// Therefore, downloading code into FLASH can take place without overwriting the present 
// application.  A new image will by activated only after the download has been verified.
code const unsigned long half_offsets[2]={0x4000L,0x20000L};

// This header pattern marks the start of the secondary configuration for the CSoC.
const unsigned char magic_pattern[4]={0x80,0x25,0xA5,0xA9};

static unsigned long flash_address;
static unsigned long magic_address; // Address at which the header was found
static unsigned long new_magic_address;

static unsigned int num_magic_markers;  // Number of headers detected in downloaded file
                                        // should be exactly 1.

static void erase_half(char half)
{
   // segment 0 is never erased because it holds essential bootstrap code.
   if (half==0) {
   	flash_erase_segment(flash_segment_offsets[1]);
   	flash_erase_segment(flash_segment_offsets[2]);
   	flash_erase_segment(flash_segment_offsets[3]);
   	flash_erase_segment(flash_segment_offsets[4]);
   } else {
   	flash_erase_segment(flash_segment_offsets[5]);
   	flash_erase_segment(flash_segment_offsets[6]);
   }  
} // erase_half

void initialise_flash() {
	char match,i,half;
	// Determine which half of Flash contains the valid download image.

	// The half of the Flash containing the current image will have the
	// header pattern, indicating the start of secondary configuration.
	// The header will appear on a 16K boundary.  The initial 16K boundary 
	// contains this startup code, so it does not need to be check.
	// The header must also occur in the last 256K of memory but the 
	// Triscend E5 development board only has 256K of Flash anyway.
   
	for (magic_address=flash_segment_offsets[1];magic_address<0x40000L;magic_address+=0x4000) {
   	match=1;
		for (i=0;i<4;i++) {
			match &= (flash_read(magic_address+i)==magic_pattern[i]);
 		}
  		if (match) break;
  	}
  	
  	// The half that does not have the header pattern is the one to erase.
  	half= !(magic_address > half_offsets[1]);
   // Erase the segments in that half of Flash.  Flash memory segments must be erased
   // before new data can be written to them.
   
   erase_half(half);
   
   flash_address=half_offsets[half];
   num_magic_markers=0;
} // initialise flash 

void write_packet(packet)
unsigned char packet[8];
{
	char i;
	unsigned char match;
	
   // Check if this packet contains the header marker.
   // Header pattern must occur on a 16K address boundary
	if ((flash_address & 0x3FFFL)==0) {
		match=1;
		for (i=0;i<4;i++) {
			match &= (packet[i]==magic_pattern[i]);
 		}
  		if (match) {
  			// DO NOT write the first byte of the header marker just yet.
  	      	        // This ensures that the 'official' header marker, which marks code
  	  		// for execution by the CSoC, will ONLY appear after the
  	   		// verification of the MAC.
  			new_magic_address=flash_address;
  	 		flash_address++; 		// Leave this location blank, temporarily
  	   	    num_magic_markers++;   // Check that the file does not have multiple header markers
  	   	 
  		  	for (i=1;i<8;i++) {
				flash_write(flash_address,packet[i]);
 				flash_address++;
  			}
  	 		return;
       }
   }
   // Its a 'normal' packet - write to the flash.
  	for (i=0;i<8;i++) {
		flash_write(flash_address,packet[i]);
 		flash_address++;
  	}
  	
} // write_packet

// If the downloaded image does not have a header marker, then the CSoC cannot run it.
// If it has more than one, then someone might be trying to overcome the download security.
unsigned char code_is_runnable() {
 return (num_magic_markers==1);
} // code is runnable

void activate() {
  char half;
  
  // Overwrite code marker on previous software
  // Find the half containing the magic_address
  half=(magic_address > half_offsets[1]);
  
  // Install the "skipped" byte of the new header BEFORE erasing the old software.  If anything
  // goes wrong, it iss better to have two sets of executable software than none!
  // At this point the MAC has been checked and we know the new software is good.
  flash_write(new_magic_address,magic_pattern[0]);
  
  // Erase the old image.  It is possible that someone could deliberately write code
  // where the header appears more than once on a 16K boundary, so erasing only 
  // the segment in which the marker was found is insufficient.
  erase_half(half);
  
  // The header for the currently executing code is now removed.  The
  // only valid header in the Flash is in the image just downloaded.
  // Consequently, the new image starts executing following a reset.
  
  // User must reset or power off the CSoC. Software cannot create a hardware level reset
  // and only a full reset will reload the CSL logic.
  while (1);
  
} // activate

//
// This routine is called when the MAC on downloaded code is incorrect.  It completely
// removes the downloaded image from Flash memory.  This is a little paranoid since the
// new software will not have a valid magic marker.  
// 
void deactivate() {
  char half;
  
  // Overwrite code header on previous software
  // Find the half containing the downloaded code
  half=!(magic_address > half_offsets[1]);
  // Erase the whole thing.  It is possible that someone could deliberately write code
  // where the header appears more than once so erasing only the segment in which
  // the marker was found is insufficient.
  
  erase_half(half);
  
  // Start writing flash from the beginning again
  flash_address=half_offsets[half];
} // deactivate
