/*
  serial.c

  This code is based on an example program written by Keil software to
  demonstrate the RTX51 O/S.  It was modified by Triscend for the E5
  processor.
  
  Changes:
  - no Ctrl-S/Ctrl-Q handshake on transfers to the PC from this code.  This
    application never sends more than a few characters to the PC.
  - deleted unused code e.g. line editor.
  - support binary transfers, no interpretation of control characters.
  - UART initialisation for 9600 baud, 8 bits, no parity, 1 stop bit.
  
  Modifications (C) 1999 Triscend, Algotronix
  
  13/10/99
  
*/

/******************************************************************************/
/*                                                                            */
/*       SERIAL.C:  Interrupt Controlled Serial Interface for RTX-51 tiny     */
/*                                                                            */
/******************************************************************************/

#include "serial.h"
#include <stdio.h>
#include <reg52.h>                    /* special function register 8052       */
#include <rtx51tny.h>                 /* RTX-51 tiny functions & defines      */

#define  OLEN  8                      /* size of serial transmission buffer   */
unsigned char  ostart;                /* transmission buffer start index      */
unsigned char  oend;                  /* transmission buffer end index        */
idata    char  outbuf[OLEN];          /* storage for transmission buffer      */
unsigned char  otask = 0xff;          /* task number of output task           */

#define  ILEN  8                      /* size of serial receiving buffer      */
unsigned char  istart;                /* receiving buffer start index         */
unsigned char  iend;                  /* receiving buffer end index           */
idata    char  inbuf[ILEN];           /* storage for receiving buffer         */
unsigned char  itask = 0xff;          /* task number of output task           */

#define   CTRL_Q  0x11                /* Control+Q character code             */
#define   CTRL_S  0x13                /* Control+S character code             */

bit   sendfull;                       /* flag: marks transmit buffer full     */
bit   sendactive;                     /* flag: marks transmitter active       */
bit   sendstop;                       /* flag: marks XOFF character           */

/******************************************************************************/
/*       putbuf:  write a character to SBUF or transmission buffer            */
/******************************************************************************/
putbuf (char c)  {
  if (!sendfull)  {                   /* transmit only if buffer not full     */
    if (!sendactive && !sendstop)  {  /* if transmitter not active:           */
      sendactive = 1;                 /* transfer the first character direct  */
      SBUF = c;                       /* to SBUF to start transmission        */
    }
    else  {                           /* otherwize:                           */
      outbuf[oend++ & (OLEN-1)] = c;  /* transfer char to transmission buffer */
      if (((oend ^ ostart) & (OLEN-1)) == 0)  sendfull = 1;
    }                                 /* set flag if buffer is full           */
  }
}


/******************************************************************************/
/*       putchar:  interrupt controlled putchar function                      */
/******************************************************************************/
char putchar (char c)  {
  if (c == '\n')  {                   /* expand new line character:           */
    while (sendfull)  {               /* wait for transmission buffer empty   */
      otask = os_running_task_id ();  /* set output task number               */
      os_wait (K_SIG, 0, 0);          /* RTX-51 call: wait for signal         */
      otask = 0xff;                   /* clear output task number             */
    }
    putbuf (0x0D);                    /* send CR before LF for <new line>     */
  }
  while (sendfull)  {                 /* wait for transmission buffer empty   */
    otask = os_running_task_id ();    /* set output task number               */
    os_wait (K_SIG, 0, 0);            /* RTX-51 call: wait for signal         */
    otask = 0xff;                     /* clear output task number             */
  }
  putbuf (c);                         /* send character                       */
  return (c);                         /* return character: ANSI requirement   */
}


/******************************************************************************/
/*       _getkey:  interrupt controlled _getkey                               */
/******************************************************************************/
char _getkey (void)  {
  while  (iend == istart)  {
    itask = os_running_task_id ();    /* set input task number                */
    os_wait (K_SIG, 0, 0);            /* RTX-51 call: wait for signal         */
    itask = 0xff;                     /* clear input task number              */
  }
  return (inbuf[istart++ & (ILEN-1)]);
}


/******************************************************************************/
/*       serial:  serial receiver / transmitter interrupt                     */
/******************************************************************************/
serial () interrupt 4 using 2  {     /* use registerbank 2 for interrupt      */
  unsigned char c;
  bit   start_trans = 0;

  if (RI)  {                         /* if receiver interrupt                 */
    c = SBUF;                        /* read character                        */
    RI = 0;                          /* clear interrupt request flag          */
    
        if (istart + (unsigned char) ILEN != iend)  {
          inbuf[iend++ & (ILEN-1)] = c;
        }
                                     /* if task waiting: signal ready         */
        if (itask != 0xFF) isr_send_signal (itask);
  }

  if (TI || start_trans)  {          /* if transmitter interrupt              */
    TI = 0;                          /* clear interrupt request flag          */
    if (ostart != oend)  {           /* if characters in buffer and           */
      if (!sendstop)  {              /* if not Control+S received             */
        SBUF = outbuf[ostart++ & (OLEN-1)];      /* transmit character        */
        sendfull = 0;                /* clear 'sendfull' flag                 */
                                     /* if task waiting: signal ready         */
        if (otask != 0xFF)  isr_send_signal (otask);
      }
    }
    else sendactive = 0;             /* if all transmitted clear 'sendactive' */
  }

}


/******************************************************************************/
/*       serial_init: initialize serial interface                             */
/******************************************************************************/
 
 // make them static so we don't get clashes at link time - these symbols
 // will also be defined in the user's header file generated by FastChip.
 static xdata char DMAP3_CTL _at_ 0xFF1B;
 static xdata char OSC_SEL _at_ 0xFE81;


void serial_init ()  {
   
	DMAP3_CTL = 0x01;	/* Assign DMAP3_CTL = 1 (Dmap3 = 4kbytes size) */
	OSC_SEL = 0x01;		/* Select external oscillator as the clock source */
	DMAP3_CTL = 0x00;	/* Assign DMAP3_CTL = 0 (Dmap3 = 256 bytes ) */


   // Set up for 9600 baud, no parity, one stop bit
	TMOD&= 0x0f;
	TMOD|= 0x020;    /* timer 1 mode 2: 8-Bit reload          */
	PCON&= 0x7f;
	TCON|= 0x40;
	TH1= 0x0f9;
	SCON= 0x50;       /* mode 1: 8-bit UART, enable receiver   */
	IE|=  0x90;

}   // serial init


