/* copycode.c
   
   Modified version of Triscend demonstration code for use in software
   download startup code.  The modifications support copying a
   32K memory block aligned on a 16K boundary.
   
 (c) Copyright 1999 by Triscend Corporation (www.triscend.com).  All Rights Reserved.

 */

/*=========================================================================
| (c) 1999  Triscend Corporation
|--------------------------------------------------------------------------
| Project : Copy Code
| File    : copycode.c
|--------------------------------------------------------------------------
| This module copies code from external memory to the internal SRAM
| and starts execution there.  This is to provide users with a low-power
| execution environment while maintaining Flash programmability.
|
| Usage:  download into external memory with bank-switching enabled.
|         Bank0 = copycode.hex,  Bank1 = <application code HEX file>
 ========================================================================*/

#include "nvdata.h"
#include "copycode.h"
#include "startup.h"


/*=========================================================================
| CONSTANTS
 ========================================================================*/
/* Maximum code size is 32KB */
#define CODE_SIZE      (32 * 1024)  /* bytes */

/* Amount of internal SRAM to reserve for XDATA */
#define XDATA_SRAM_KB  4  /* KB */


/*=========================================================================
| EXTERNS
 ========================================================================*/
extern void jumpToRom ();




/*=========================================================================
| SPECIAL REGISTERS
 ========================================================================*/
static xdata volatile byte CMAP1_TAR_0  _at_ 0xff02;
static xdata volatile byte CMAP1_TAR_1  _at_ 0xff03;
static xdata volatile byte CMAP1_TAR_2  _at_ 0xff04;
static xdata volatile byte CMAP1_SRC    _at_ 0xff05;
static xdata volatile byte CMAP1_CTL    _at_ 0xff06;
static xdata volatile byte CMAP1_ALT    _at_ 0xff07;

static xdata volatile byte CMAP2_TAR_0  _at_ 0xff08;
static xdata volatile byte CMAP2_TAR_1  _at_ 0xff09;
static xdata volatile byte CMAP2_TAR_2  _at_ 0xff0a;
static xdata volatile byte CMAP2_SRC    _at_ 0xff0b;
static xdata volatile byte CMAP2_CTL    _at_ 0xff0c;
static xdata volatile byte CMAP2_ALT    _at_ 0xff0d;

static xdata volatile byte DMAP1_TAR_0  _at_ 0xff0f;
static xdata volatile byte DMAP1_TAR_1  _at_ 0xff10;
static xdata volatile byte DMAP1_TAR_2  _at_ 0xff11;
static xdata volatile byte DMAP1_SRC    _at_ 0xff12;
static xdata volatile byte DMAP1_CTL    _at_ 0xff13;

static xdata volatile byte DMAP2_TAR_0  _at_ 0xff14;
static xdata volatile byte DMAP2_TAR_1  _at_ 0xff15;
static xdata volatile byte DMAP2_TAR_2  _at_ 0xff16;
static xdata volatile byte DMAP2_SRC    _at_ 0xff17;
static xdata volatile byte DMAP2_CTL    _at_ 0xff18;

static xdata volatile byte DMAP4_TAR_0  _at_ 0xff80;
static xdata volatile byte DMAP4_TAR_1  _at_ 0xff81;
static xdata volatile byte DMAP4_TAR_2  _at_ 0xff82;
static xdata volatile byte DMAP4_SRC    _at_ 0xff83;
static xdata volatile byte DMAP4_CTL    _at_ 0xff84;

// Data mapper 5 is used by nvdata.c code as well
xdata volatile byte DMAP5_TAR_0  _at_ 0xff85;
xdata volatile byte DMAP5_TAR_1  _at_ 0xff86;
xdata volatile byte DMAP5_TAR_2  _at_ 0xff87;
xdata volatile byte DMAP5_SRC    _at_ 0xff88;
xdata volatile byte DMAP5_CTL    _at_ 0xff89;


/*=========================================================================
| MEMORY LOCATIONS
 ========================================================================*/
static code  volatile byte gExternalMemory[CODE_SIZE]  _at_  0x8000;
static xdata volatile byte gInternalMemory[CODE_SIZE]  _at_  0x0000;


/*=========================================================================
| MAIN PROGRAM
 ========================================================================*/
void copy_code_to_ram (FlashAddr flash_addr,unsigned int length)
{
    FlashAddr addr;
    unsigned int i;

    /* Setup the highest priority code mapper to point to
       the code in the external memory.

       0x810000 - 0x817fff  ==>  CODE[0x8000 - 0xffff]
    */
    addr= flash_addr+FLASH_MEMORY_BASE_ADDRESS;  // Form 32 bit absolute address
    flash_addr=addr;
    addr=(addr>>8);               // Throw away bottom byte - must be on 256 byte boundary
    CMAP2_TAR_0 = (addr & 0xFF);  // bits 8..15
    addr=(addr>>8);
    CMAP2_TAR_1 = (addr & 0xFF);  // bits 16..23
    addr=(addr>>8);
    CMAP2_TAR_2 = (addr & 0xFF);  // bits 24..31
    CMAP2_SRC   = 0x80;
    CMAP2_ALT   = 0x00;
    CMAP2_CTL   = 0x2e;           // 16K byte area
    

    /* Setup the highest priority data mapper to point to
       the internal memory.

       0x10000 - 0x17fff  ==>  XDATA[0x0000 - 0x7fff]
    */
    DMAP5_TAR_0 = 0x00;
    DMAP5_TAR_1 = 0x01;
    DMAP5_TAR_2 = 0x00;
    DMAP5_SRC   = 0x00;
    DMAP5_CTL   = 0x2f;          // 32K byte area
    

    // The copying code is a little complex because it needs to deal
    // with up to 32K bytes of code in Flash which is aligned on a 16K byte boundary.
    // This means it is necessary to do the 32K byte transfer as two 16K byte chunks.
    if (length>0x4000) {
	    /* Copy the first 16K chunk of code from external to internal memory */
   	for (i = 0; i < 0x4000; ++i)
       	gInternalMemory[i] = gExternalMemory[i];
        // Map in the next 16K chunk in external memory
        addr=flash_addr+0x4000;
        addr=(addr>>8);               // Throw away bottom byte - must be on 256 byte boundary
    	CMAP2_TAR_0 = (addr & 0xFF);  // bits 8..15
    	addr=(addr>>8);
    	CMAP2_TAR_1 = (addr & 0xFF);  // bits 16..23
    	addr=(addr>>8);
    	CMAP2_TAR_2 = (addr & 0xFF);  // bits 24..31
        /* Copy the second 16K chunk from external to internal memory */
      	for (i = 0; i < (length-0x4000); ++i)
       	    gInternalMemory[i+0x4000] = gExternalMemory[i];
    }  else {
     	/* Simple case, less than 16K of user code: copy the code from external to internal memory */
    	for (i = 0; i < length; ++i)
       	    gInternalMemory[i] = gExternalMemory[i];
    }

   
    /* Setup the data mappers for the appropriate amount of
       SRAM in XDATA space.

       Note: DMAP1 is used for CSL, so it is simply left alone.
    */
    DMAP5_CTL   = 0x00;
    DMAP4_CTL   = 0x00;
    DMAP2_CTL   = 0x00;

#if XDATA_SRAM_KB == 1
    /* 0x13c00 - 0x13fff  ==>  XDATA[0x0000 - 0x03ff] */
    DMAP2_TAR_0 = 0x3c;
    DMAP2_TAR_1 = 0x01;
    DMAP2_TAR_2 = 0x00;
    DMAP2_SRC   = 0x00;
    DMAP2_CTL   = 0x2a;
#endif

#if XDATA_SRAM_KB == 2
    /* 0x13800 - 0x13fff  ==>  XDATA[0x0000 - 0x07ff] */
    DMAP2_TAR_0 = 0x38;
    DMAP2_TAR_1 = 0x01;
    DMAP2_TAR_2 = 0x00;
    DMAP2_SRC   = 0x00;
    DMAP2_CTL   = 0x2b;
#endif

#if XDATA_SRAM_KB == 4
    /* 0x13000 - 0x13fff  ==>  XDATA[0x0000 - 0x0fff] */
    DMAP2_TAR_0 = 0x30;
    DMAP2_TAR_1 = 0x01;
    DMAP2_TAR_2 = 0x00;
    DMAP2_SRC   = 0x00;
    DMAP2_CTL   = 0x2c;
#endif

   
    /* Copy the lower priority code mapper to the higher priority
       mapper so that the lower priority mapper can be programmed
       with internal SRAM's information.
    */
    CMAP2_TAR_0 = CMAP1_TAR_0;
    CMAP2_TAR_1 = CMAP1_TAR_1;
    CMAP2_TAR_2 = CMAP1_TAR_2;
    CMAP2_SRC   = CMAP1_SRC;
    CMAP2_ALT   = CMAP1_ALT;
    CMAP2_CTL   = 0x2f;         /* Only enable 32KB so that the
                                   built-in ROM shows through in
                                   the upper half of code space.
                                */
    
    /* Program the lower priority mapper for the internal SRAM */
    CMAP1_TAR_0 = 0x00;
    CMAP1_TAR_1 = 0x01;
    CMAP1_TAR_2 = 0x00;
    CMAP1_SRC   = 0x00;
    CMAP1_ALT   = CMAP2_ALT;
    CMAP1_CTL   = 0x00;

  
    /* Jump to a location in the built-in ROM code which will
       switch mappers and roll over into the user code at CODE[0x0000].
    */
    jumpToRom();
} // copy_code
