/*=========================================================================
 nvdata.c
 
(c) Copyright 1999 by Triscend Corporation (www.triscend.com).  All Rights Reserved.
  
 ========================================================================*/

/*=========================================================================
| INCLUDES
 ========================================================================*/
#include "nvdata.h"
#include "copycode.h"


/*=========================================================================
| FLASH CONSTANTS
 ========================================================================*/
#define FLASH_MEMORY_EXP      13      /* 2^13 = 8KB, must be >= 8   */
#define FLASH_MEMORY_SIZE     (1 << FLASH_MEMORY_EXP)
#define FLASH_MEMORY_MASK     (0xffffffffL << FLASH_MEMORY_EXP)


/*=========================================================================
| NV DATA LOCATION
 ========================================================================*/
/* Define the location of the NV data in the Flash.
   Uncomment the appropriate definition below.
*/
/**** AM29LV002BB, AM29LV001BB, AM29LV001BT ****/
#define NV_DATA_LOCATION      0x08000L

/**** AM29LV002BT ****/
// #define NV_DATA_LOCATION      0x3c000L

/**** USER DEFINED ****/
// #define NV_DATA_LOCATION      0x00000L


/*=========================================================================
| TYPEDEFS
 ========================================================================*/



/*=========================================================================
| SPECIAL REGISTERS
 ========================================================================*/



/*=========================================================================
| GLOBAL VARIABLES
 ========================================================================*/
/* gFlashRam[0] corresponds to this offset in the Flash */
static FlashAddr  gFlashOffset = 0;




/*=========================================================================
| MEMORY LOCATIONS
 ========================================================================*/
static xdata volatile byte gFlashRam[FLASH_MEMORY_SIZE]  _at_  0x8000;


/*=========================================================================
| STATIC FUNCTIONS
 ========================================================================*/

/*=========================================================================
| flashMap
|--------------------------------------------------------------------------
| Map the Flash so that it includes the flash_addr.
| This function is intelligent in the sense that it does not remap
| the Flash if the current block already includes the address.
 ========================================================================*/
static void
flashMap (FlashAddr flash_addr)
{
    static bit first = 1;

    /* If the address is already included in the block, return.
       If this is the first time, force a mapping.
    */
    if (first                               == 0          &&
        gFlashOffset                        <= flash_addr &&
        gFlashOffset +FLASH_MEMORY_SIZE -1  >= flash_addr)
        return;

    first = 0;


    /* Map in the Flash, which sits at 32-bit address 0x800000 */
    flash_addr  &= FLASH_MEMORY_MASK;
    gFlashOffset = flash_addr;
    flash_addr  += FLASH_MEMORY_BASE_ADDRESS;

    /* Disable it first */
    DMAP5_CTL   = 0x00;

    /* Map the Flash address */
    DMAP5_TAR_2 = (flash_addr >> 24) & 0xff;
    DMAP5_TAR_1 = (flash_addr >> 16) & 0xff;
    DMAP5_TAR_0 = (flash_addr >>  8) & 0xff;

    /* Map it to 0x8000 in the XDATA space */
    DMAP5_SRC   = 0x80;

    /* Enable it and set the size to that of the data buffer */
    DMAP5_CTL   = 0x20 | FLASH_MEMORY_EXP;
}  // flashMap


/*=========================================================================
| flashRead
| flashWrite
|--------------------------------------------------------------------------
| Read or write a byte from the Flash.
 ========================================================================*/
static void flash_write_control (FlashAddr addr, byte value)
{
    flashMap (addr);
    gFlashRam[addr - gFlashOffset] = value;
} // flash_write_control

byte flash_read (FlashAddr addr)
{
    /* Reset the Flash */
    flash_write_control (0, 0xf0);

    flashMap (addr);
    return gFlashRam[addr - gFlashOffset];
} // flash_read


  
