/*------------------------------------------------------------------------------

startup.c

(c) Copyright 1999 by Triscend Corporation (www.triscend.com).  All Rights Reserved.


After a reboot, this code transfers the main software image from FLASH into 
the on-chip SRAM available in an E520 CSoC .

------------------------------------------------------------------------------*/


#include "nvdata.h"
#include "copycode.h"

// There is 40K bytes of on-chip RAM on the E520 CSoC.  Assume that a 
// maximum of 32K can be used for code.
#define MAX_AVAILABLE_RAM 0x8000

// When the Triscend FastChip software is used to download banked code it
// puts the second bank at this address (64K).
#define INITIAL_CODE_LOCATION 0x10000L

// Offsets of segments in AMD 29LV0002BB chip on the Triscend E5 Development Board.
// Taken from Table 3. in AM29LV002B data sheet.
code const long flash_segment_offsets[7]={
	0L,   // Segment 0 holds constant (non-downloaded) bootsrap code
   0x4000L,0x6000L,0x8000L,0x10000L, // Half 0, 112K for downloaded code, in 4 segments
   0x20000L,0x30000L                 // Half 1, 128K for downloaded code, in 2, 64K segments
   };

// If the E520 CSoC booted from Half 0, then it will download a new initialization image
// Therefore, downloading code  into FLASH can take place without overwriting the present
// application.  The new image will be activated only after the download has been verified.
code const long half_offsets[2]={0x4000L,0x20000L};

// This header pattern marks the start of the secondary configuration for the CSoC.
code const unsigned char magic_pattern[4]={0x80,0x25,0xA5,0xA9};

static long start_address;   // address of the start of 8032 code
static long magic_address;   // address the header marker was found

void main(void) {
	unsigned char match,i,half;
	unsigned int length;
	//unsigned long l;

	// Determine which half of Flash contains the valid download image.

	// The half of the Flash containing the current image will have the
	// header pattern, indicating the start of secondary configuration.
	// The header will appear on a 16K boundary.  The initial 16K boundary 
	// contains this startup code, so it does not need to be check.
	// The header must also occur in the last 256K of memory but the 
	// Triscend E5 development board only has 256K of Flash anyway.
   
   
  	for (magic_address=flash_segment_offsets[1];magic_address<0x40000L;magic_address+=0x4000) {
   	match=1;
		for (i=0;i<4;i++) {
			match &= (flash_read(magic_address+i)==magic_pattern[i]);
 		}
  		if (match) break;
  	}
  	// The half of Flash that contains the header pattern also contains the 
	// code to load into SRAM.
  	// The CSoC will already have already loaded the CSL configuration from this half of the FLASH.
  	half= (magic_address > half_offsets[1]);


   start_address=half_offsets[half];
   // When the Flash is initially programmed by the FastChip software, the user code starts
   // at 64K in Flash, instead of at 16K.  This is because FastChip's Flash download program
   // was designed to load 64K code banks, not individual .hex files.

   // Consequently, the first download of software (after loading the initial image using 
   // FastChip) is a special case and will overwrite part of the Flash containing the
   // currently  running code.
   if ((half==0)&&(magic_address > INITIAL_CODE_LOCATION)) {
   	start_address=INITIAL_CODE_LOCATION;
   }

   // Find end of code
   // Secondary initialization starts on a 16K byte boundary following end of code (at magic_address)
   // Copy an integral number of 16K byte blocks rather than attempting to find
   // the actual last byte of code.
   length=magic_address-start_address;

   if (length<=MAX_AVAILABLE_RAM) {
		// Copy code into SRAM and jump into it
		copy_code_to_ram(start_address,length);
 		// copy_code_to_ram does not return, control is passed to the user application.
   } else {

	// Code was too large to copy.  What action to take depends on the end application.
   }

   while (1);
} // main


