/*------------------------------------------------------------------------------

  DES.C

  (c) Copyright 1999 by Triscend Corporation (www.triscend.com).
  All Rights Reserved.

  This code implements Triple DES (Encrypt,Decrypt,Encrypt EDE) in Cipher 
  Block Chaining (CBC) mode with a Message Authentication Code (MAC)
  for the encrypted file.

  The code is optimized for the 8051 8-bit microcontroller in that 8 bit quantities
  have been used wherever possible.

------------------------------------------------------------------------------*/



#include <stdio.h>                /* prototype declarations for I/O functions */
#include "des.h"
/* On the PC define code as a null string - the code directive is
   8051 specific. */
#define code



/* 
DES Bit Numbering

The DES standard numbers bits such that bit 1 is MSB and bit 64 is LSB in
a 64 bit input word!  This is very different from the normal bit numbering 
in C.  However, changing to the C standard has the big disadvantage that all
the DES tables need to be recoded and the description in the various textbooks
becomes hard to relate to the code.

So this program sticks with the DES conventions  EXCEPT when a 32 bit or
64 bit quantity is loaded into an array of bytes the bytes are numbered
0 through 7 not 1 through 8.  Byte 0 holds is Most significant.  This change
from the DES numbering allowed more efficient code.  

*/

/* The DES S-Boxes.  S-Box constants checked  8/9/99 against Stalling's book. */

code const unsigned char s[8][64]=
 /* S1 */              {14,  4, 13,  1,  2, 15, 11,  8,  3, 10,  6, 12,  5,  9,  0,  7,
                         0, 15,  7,  4, 14,  2, 13,  1, 10,  6, 12, 11,  9,  5,  3,  8,
                         4,  1, 14,  8, 13,  6,  2, 11, 15, 12,  9,  7,  3, 10,  5,  0,
                        15, 12,  8,  2,  4,  9,  1,  7,  5, 11,  3, 14, 10,  0,  6, 13,
/* S2 */ 
                        15,  1,  8, 14,  6, 11,  3,  4,  9,  7,  2, 13, 12,  0,  5, 10,
                         3, 13,  4,  7, 15,  2,  8, 14, 12,  0,  1, 10,  6,  9, 11,  5,
                         0, 14,  7, 11, 10,  4, 13,  1,  5,  8, 12,  6,  9,  3,  2, 15,
                        13,  8, 10,  1,  3, 15,  4,  2, 11,  6,  7, 12,  0,  5, 14,  9,
/* S3 */
                        10,  0,  9, 14,  6,  3, 15,  5,  1, 13, 12,  7, 11,  4,  2,  8,
                        13,  7,  0,  9,  3,  4,  6, 10,  2,  8,  5, 14, 12, 11, 15,  1,
                        13,  6,  4,  9,  8, 15,  3,  0, 11,  1,  2, 12,  5, 10, 14,  7,
                         1, 10, 13,  0,  6,  9,  8,  7,  4, 15, 14,  3, 11,  5,  2, 12,
/* S4 */
                         7, 13, 14,  3,  0,  6,  9, 10,  1,  2,  8,  5, 11, 12,  4, 15,
                        13,  8, 11,  5,  6, 15,  0,  3,  4,  7,  2, 12,  1, 10, 14,  9,
                        10,  6,  9,  0, 12, 11,  7, 13, 15,  1,  3, 14,  5,  2,  8,  4,
                         3, 15,  0,  6, 10,  1, 13,  8,  9,  4,  5, 11, 12,  7,  2, 14,
/* S5 */
                         2, 12,  4,  1,  7, 10, 11,  6,  8,  5,  3, 15, 13,  0, 14,  9,
                        14, 11,  2, 12,  4,  7, 13,  1,  5,  0, 15, 10,  3,  9,  8,  6,
                         4,  2,  1, 11, 10, 13,  7,  8, 15,  9, 12,  5,  6,  3,  0, 14,
                        11,  8, 12,  7,  1, 14,  2, 13,  6, 15,  0,  9, 10,  4,  5,  3,
/* S6 */
                        12,  1, 10, 15,  9,  2,  6,  8,  0, 13,  3,  4, 14,  7,  5, 11,
                        10, 15,  4,  2,  7, 12,  9,  5,  6,  1, 13, 14,  0, 11,  3,  8,
                         9, 14, 15,  5,  2,  8, 12,  3,  7,  0,  4, 10,  1, 13, 11,  6,
                         4,  3,  2, 12,  9,  5, 15, 10, 11, 14,  1,  7,  6,  0,  8, 13,
/* S7 */
                         4, 11,  2, 14, 15,  0,  8, 13,  3, 12,  9,  7,  5, 10,  6,  1,
                        13,  0, 11,  7,  4,  9,  1, 10, 14,  3,  5, 12,  2, 15,  8,  6,
                         1,  4, 11, 13, 12,  3,  7, 14, 10, 15,  6,  8,  0,  5,  9,  2,
                         6, 11, 13,  8,  1,  4, 10,  7,  9,  5,  0, 15, 14,  2,  3, 12,
/* S8 */
                        13,  2,  8,  4,  6, 15, 11,  1, 10,  9,  3, 14,  5,  0, 12,  7,
                         1, 15, 13,  8, 10,  3,  7,  4, 12,  5,  6, 11,  0, 14,  9,  2,
                         7, 11,  4,  1,  9, 12, 14,  2,  0,  6, 10, 13, 15,  3,  5,  8,
                         2,  1, 14,  7,  4, 10,  8, 13, 15, 12,  9,  0,  3,  5,  6, 11};

/* P box constants checked 8/9/99 against Stallings book */
code const unsigned char p[32]={16,7,20,21,
                         29,12,28,17,
                         1,15,23,26,
                         5,18,31,10,
                         2,8,24,14,
                         32,27,3,9,
                         19,13,30,6,
                         22,11,4,25};

/* We are running triple DES so we require 3 independant 56 bit DES keys: key1,key2,key3.
   The DES algorithm specifies 16 'rounds' each of which uses a slightly different 48 bit
   key which is calculated from the 56 bit key.  Since the DES key in this application is
   embedded in the equipment and not changed instead of calculating the round keys at
   'run time' we pre-calculate them and store the round keys rather than the main key
   in these tables.  This reduces the run time for DES, the extra storage is compensated
   for by the fact that the key handling code is no longer required. 

   Only the bottom 6 bits in each byte of these tables is significant (8*6=48 bits) 

   Triple DES in EDE mode.
   Data -> Encrypt(key1) -> Decrypt(key2) -> Encrypt(key3)   
   If the user sets key1==key2==key3 then this is the same as Encrypt(key1)  */
code static unsigned char key1[16][8];
code static unsigned char key2[16][8];
code static unsigned char key3[16][8];

static unsigned char keyl[4], keyr[4];

/* left and right are 32 bit numbers corresponding to L and R in Figure 12.2 of Schneier
   the low nibble of each byte in the array holds 4 bits of the key. Element 0 of the
   array is most significant.  This sparse format is used to make implementing the
   expansion permutation more efficient. */
static unsigned char left[8],right[8];

void des_round(round_key)
unsigned char *round_key;
{

  unsigned char i,exp,out[8],sout[8];
  unsigned char above,below,pb;
  char b,x;  /* NB. this loop counter must be signed to allow >=0 comparison */

  //printf("Des Round\n");
  for (i=0;i<8;i++) {out[i]=0;}
  
  for (i=0;i<8;i++) {
    /* Implement the expansion permutation E.  See fig 12.3 on page 274 of Schneier.
     The variable 'right' is a 32 bit word split up into 8 bytes, each byte contains
     one nibble of the word.  Byte 0 is the Most Significant!  See the note on DES
     bit numbering at the top of this file. */
    above=((i==0)? 7:i-1);  /* above and below wrap around at the end of the array */
    below=((i==7)? 0:i+1);
    exp=((right[above] & 0x01)? 0x20:0) | (right[i]<<1) | ((right[below] & 0x08)!=0);
	
    /* XOR with key - exp is a 6 bit number*/
    x=exp^round_key[i];
	
	exp=((x & 0x1E)>>1) | (x & 0x20) | ((x & 0x01)<<4);
    /* Implement S-box substitution */
    sout[i]=s[i][exp];
	
  }
  
  /* got S box output in sout[].  Now need to do the P box permutation */
  for (b=0;b<32;b++) {
      /* Use the P box permutation table to find the bit position in the output where this bit goes */
      /* Convert from 1..32 format in the table to 0..31, to make indexing easier. */
	  pb=p[b]-1;
	  x=( (sout[pb>>2] & (1<<(3-(pb & 0x03)))) !=0);
      /* and put it in.  Need to figure out which byte it is in */
      out[b>>2]=out[b>>2] | (x<<(3-(b & 0x03)));
  }
  
  /* XOR L with P box output and swap L,R over.  See fig 12.2 in Schneier */
  for (i=0;i<8;i++) {
    out[i]=left[i] ^ out[i];
    left[i]=right[i];
    right[i]=out[i];
  }
} /* des_round */

void des_encrypt(key)
unsigned char key[][8];
{
   char round;
   unsigned char *round_key;
   
  //printf("Des Encrypt\n");
   for (round=0;round<16;round++) {
      round_key=key[round];
	  
      des_round(round_key);
   }
} /* des_encrypt */

/* DES decryption is the same as encryption except that the round keys
   are applied in the opposite order */
void des_decrypt(key)
unsigned char key[][8];
{
   char round;
   unsigned char * round_key;
   
   for (round=15;round>=0;--round) {
      round_key=key[round];
      des_round(round_key);
   }
} /* des_decrypt */

void des_ip(plain)
unsigned char plain[];
{
  char i,j,x;
  unsigned char plain_buf[8],rot[8];
  //printf("Des IP\n");
 
  // Make a working copy of the input parameter - need to shift it about
  // and don't want to mess up external version.
  for (i=0;i<8;i++) {
	  plain_buf[i]=plain[i];
  }
  /* Rotate the input data so all the MSB's end up in byte 0, all the LSB's in byte 7 */
  for (i=0; i<8;i++) {
    rot[i]=0;
    for (j=7;j>=0;j--) {
      x=((plain_buf[j] & 0x80) !=0); /* get MSB as a boolean without shifting */
      plain_buf[j]=plain_buf[j]<<1;
      rot[i]=(rot[i]<<1) | x;
    }
  }
   
  /* Even numbered bytes go into 'left', odd numbered bytes go into 'right' */
  for (i=0;i<4;i++) {
    left[2*i]=(rot[i*2+1]>>4);
	left[2*i+1]=(rot[i*2+1] & 0x0F);
    right[2*i]=(rot[i*2]>>4);
    right[2*i+1]=(rot[i*2] & 0x0F);
  }
   
} /* des_ip */



/* The DES final permutation is the inverse of the initial permutation.  The
   code is the same as for des_ip except 'the other way round'. */
void des_fp(cipher)
unsigned char cipher[];
{
  char i,j,x;
  unsigned char rot[8];
 
  /* printf("Des FP\n");
  printf("before FP=");
     for (i=0;i<8;i++) {
      printf("%x ",right[i]);
     }; 
	 for (i=0;i<8;i++) {
      printf("%x ",left[i]);
     }; 
   printf("\n");
  */
  /* N.B.  See fig 12.1 in Schneier.  Everywhere else left is MS word and right is LS word
     however after final round of DES going into the FP right is MS and left is LS.
     Odd numbered bytes come from 'right', even numbered bytes come from 'left' 
	 left and right are not dense - only first nibble is significant, need to reconstruct
	 full bytes. */
  for (i=0;i<4;i++) {
	rot[i*2]=(left[2*i]<<4) | left[2*i+1];
    rot[i*2+1]=(right[2*i]<<4) | right[2*i+1];
 
  }
  /* Rotate the input data so all the LSB's end up in byte 0, all the MSB's in byte 7 */
  for (i=0; i<8;i++) {
	cipher[i]=0;
    for (j=0;j<8;j++) {
	  x=((rot[j] & 0x01) !=0); /* get LSB as a boolean without shifting */
      rot[j]=rot[j]>>1;
	  cipher[i]=(cipher[i]<<1) | x;
    }
  } 
 
} /* des_fp */

void des_key_perm(key)
unsigned char key[];
{
  char i,j,x;
  unsigned char rot[8];
  char lprev,rprev;
                               
  //printf("Des key perm\n");
  /* The table for this permutation is in Table 12.2 of Schneier.  This code
   uses the structure of the permutation to get a more efficient implementation. */
  /* Rotate the input data so all the MSB's end up in byte 0, all the LSB's in byte 7 */
  for (i=0; i<7;i++) {
    rot[i]=0;
    for (j=7;j>=0;j--) {
      x=((key[j] & 0x80) !=0); /* get MSB as a boolean without shifting */
      key[j]=key[j]<<1;
      rot[i]=(rot[i]<<1) | x;
    }
  }
  
  /* Bits 64,56,48,40,32,24,16,8 are parity and get thrown away - these are in rot[7]. 
     The remaining 56 bits get broken into two 28 bit keys, keyl is most significant,
     keyr least significant.  The top 4 bits of keyl[0] and keyr[0] are 0.  The top
     nibble of rot[3] ends up as the LS nibble of keyl[3], the bottom nibble of rot[3]
     ends up as the LS nibble of keyr[3]. */
  lprev=0; rprev=0;
  for (i=0;i<4;i++) {
    keyl[i]=(rot[i]>>4) | (lprev<<4);
    lprev=rot[i];
    keyr[i]=(rot[6-i]>>4) | (rprev<<4);
	rprev=rot[6-i];
  }
  
} /* des_key_perm */

/* Number of key shifts on each round
                          Round  1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 */
code const char round_shifts[16]={1,1,2,2,2,2,2,2,1, 2, 2, 2, 2, 2, 2, 1};

/* Key compression permutation */
code const char comp_perm[48]={14,17,11,24, 1, 5, 3,28,15, 6,21,10,
                               23,19,12, 4,26, 8,16, 7,27,20,13, 2,
                               41,52,31,37,47,55,30,40,51,45,33,48,
                               44,49,39,56,34,53,46,42,50,36,29,32};

/* Set up precalculated table of round keys for use by encryption/decryption
   algorithms based on 28 bit keyl and keyr.  In software it is much more efficient to
   calculate the 16 round keys only when the key is changed  and store them in a table. */
void des_key_sched(round_key)
unsigned char round_key[][8];
{
   char s,i;
   unsigned char lwrap,rwrap,round,ob,x;

   //printf("Des key sched\n");
   for (round=0;round<16;round++) {
     for (s=0;s<round_shifts[round];s++) {
       /* Shifts are rotate left. */
       lwrap=((keyl[0] & 0x08) !=0);  /* MS bit -- its a 28 bit number */
       rwrap=((keyr[0] & 0x08) !=0);
       for (i=0;i<4;i++) {
         keyl[i]=(keyl[i]<<1)|((i!=3)?((keyl[i+1]&0x80)!=0):lwrap);
         keyr[i]=(keyr[i]<<1)|((i!=3)?((keyr[i+1]&0x80)!=0):rwrap);
       }
     }
     
	 
     /* After the shifting cut keys back down to 28 bits. */
     keyl[0]=keyl[0] & 0x0F; keyr[0]=keyr[0] & 0x0F;
	 
	 for (i=0;i<8;i++) {
		 round_key[round][i]=0;
	 }
     /* Next we need to apply the compression permutation to convert 56 bit key to
        48 bit round key and store the round key in the table ready for use.
        There is no obvious structure to the compression permutation so implement it
        straightforwardly with lots of bit operations. 
	    The bit indexing code is nasty because keyl and keyr are 28 bit numbers stored
	    in 4 bytes of memory - i.e. the top nibble of keyl[0] and keyr[0] is not part
	    of the number.  This made it easier to count bits starting at the LSB for the
	    purposes of indexing.  */
     for (i=0;i<48;i++) {
       ob=56-comp_perm[i];  /* Convert 1..56 format used in table to normal MSB 55..0 LSB */
	   if (ob>=28) { /* ob is in keyl */
	     ob=ob-28;
         x=( (keyl[3-(ob>>3)] & (1<<(ob & 0x07)))!=0);  /* extract bit 'ob' from keyl*/
       } else {
		 x=( (keyr[3-(ob>>3)] & (1<<(ob & 0x07)))!=0); 
	   }
	   round_key[round][i/6]=round_key[round][i/6] | (x<<(5-(i%6)));
     }
	 
   }
} /* des_key_sched */


void des_ecb(plain,cipher)
unsigned char plain[8],cipher[8];
{
   const unsigned char key[8]=  {0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef};
   //const unsigned char plain[8]={0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xe7};
 
   //printf("Des ECB\n");
   des_key_perm(key);
   
   des_key_sched(key1);
   des_ip(plain);
   des_encrypt(key1);
   
   des_fp(cipher);
   /*for (i=0;i<8;i++) {
     printf("%x ",cipher[i]);
   }
   printf("\nCorrect Value c9 57 44 25 6a 5e d3 1d\n");
   */
}  /* des_ecb */

/* Implement Triple DES in the Encrypt,Decrypt,Encrypt (EDE) mode */
void des3_ede(plain,cipher)
unsigned char plain[8],cipher[8];
{
   // If KeyA=KeyB=KeyC then Encrypt(KeyA),Decrypt(KeyB),Encrypt(KeyC) is
   // equivalent to Encrypt(KeyC).
   // This allows using this function to implement single DES encryption.

   // Least significant bits of each byte are a parity check in the standard
   // This code does not check parity  since the key is not likely to
   // be corrupted so they are ignored.
   const unsigned char keyA[8]=  {0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef};
   const unsigned char keyB[8]=  {0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef};
   const unsigned char keyC[8]=  {0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef};
   //const unsigned char plain[8]={0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xe7};
   unsigned char t[8];
   char i;
  /* In theory every des_encrypt should implement the initial and final
     permutations.  But an IP immediately followed by an FP cancels out
     so we only do it at the beginning and end of the chain. */
  des_key_perm(keyA);  
  des_key_sched(key1);
  des_key_perm(keyB);  
  des_key_sched(key2);
  des_key_perm(keyC);  
  des_key_sched(key3);

  des_ip(plain);
  des_encrypt(key1);
  // this swap would normally happen in FP.
  for (i=0;i<8;i++) { t[i]=left[i];left[i]=right[i];right[i]=t[i]; }
  des_decrypt(key2);
  for (i=0;i<8;i++) { t[i]=left[i];left[i]=right[i];right[i]=t[i]; }
  des_encrypt(key3);
  des_fp(cipher);  
  //for (i=0;i<8;i++) {
  //   printf("%x ",cipher[i]);
  //}
  //printf("\nCorrect Value c9 57 44 25 6a 5e d3 1d\n");
} /* des3_ede */


/* Implement Triple DES Decryption */
void des3_ede_decrypt(cipher,plain)
unsigned char plain[8],cipher[8];
{
   const unsigned char key[8]=  {0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef};
   // This is a standard DES test vector.
   //const unsigned char plain[8]={0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xe7};
   unsigned char t[8];
   char i;
  /* In theory every des_encrypt should implement the initial and final
     permutations.  But an IP immediately followed by an FP cancels out
     so we only do it at the beginning and end of the chain. */
  des_key_perm(key); 
  des_key_sched(key1);
  des_key_perm(key); 
  des_key_sched(key2);
  des_key_perm(key); 
  des_key_sched(key3);
  des_ip(cipher);
  des_decrypt(key1);
  // this swap would normally happen in FP.
  for (i=0;i<8;i++) { t[i]=left[i];left[i]=right[i];right[i]=t[i]; }
  des_encrypt(key2);
  for (i=0;i<8;i++) { t[i]=left[i];left[i]=right[i];right[i]=t[i]; }
  des_decrypt(key3);
  des_fp(plain);  
  // Test code - check the encryption of plain[] using key[].
  //for (i=0;i<8;i++) {
  //   printf("%x ",cipher[i]);
  //}
  //printf("\nCorrect Value c9 57 44 25 6a 5e d3 1d\n");
} /* des3_ede_decrypt */

