/*------------------------------------------------------------------------------

  datagram.c

  (c) Copyright 1999 by Triscend Corporation (www.triscend.com).
  All Rights Reserved.

  This code creates a reliable serial communications channel to transfer
  large files between a PC and the Triscend evaluation board.

------------------------------------------------------------------------------*/



#include <stdlib.h>                /* prototype declarations for I/O functions */
#include <stdio.h>
#include <ctype.h>
#include <windows.h>

#include "datagram.h"

static HANDLE serial_port;


static int chars_sent=0;
static int datagrams_sent=0;
unsigned char download_status=IN_PROGRESS;

// Microsoft example code to set up serial interface.
static void configure_serial(void) {

	DCB dcb;
	DWORD dwError;
	BOOL fSuccess;
	COMMTIMEOUTS timeouts;

	chars_sent=0;

	serial_port = CreateFile("COM2",
		GENERIC_READ | GENERIC_WRITE,
		0,    /* comm devices must be opened w/exclusive-access */
		NULL, /* no security attrs */
		OPEN_EXISTING, /* comm devices must use OPEN_EXISTING */
		0,    /* not overlapped I/O */
		NULL  /* hTemplate must be NULL for comm devices */
		);

	if (serial_port == INVALID_HANDLE_VALUE) {
		dwError = GetLastError();
        printf("Fatal Error %d, opening serial port\n",dwError);
		exit(1);
		/* handle error */
	}

	/*
	* Omit the call to SetupComm to use the default queue sizes.
	* Get the current configuration.
	*/

	fSuccess = GetCommState(serial_port, &dcb);

	if (!fSuccess) {
		/* Handle the error. */
		printf("Can't get state of Serial Port\n");
		exit(1);
	}
	
	/* Fill in the DCB: baud=9600, 8 data bits, no parity, 1 stop bit. */

	dcb.BaudRate = 9600;
	dcb.ByteSize = 8;
	dcb.Parity = NOPARITY;
	dcb.StopBits = ONESTOPBIT;

	fSuccess = SetCommState(serial_port, &dcb);
    
	if (!fSuccess) {
		/* Handle the error. */
		printf("Can't set state of Serial Port\n");
		exit(1);
	}

	// Tell us when a character is received.
	fSuccess= SetCommMask(serial_port,EV_RXCHAR);
  
    if (!fSuccess) {
		/* Handle the error. */
		printf("Can't set Comm Mask\n");
		exit(1);
	}

	fSuccess=GetCommTimeouts(
      serial_port, //HANDLE  hFile,	// handle of communications device  
      &timeouts    // LPCOMMTIMEOUTS  lpCommTimeouts	// address of comm. time-outs structure  
    );
    if (!fSuccess) {
		/* Handle the error. */
		printf("Can't get Comm Timeout\n");
		exit(1);
	}
	timeouts.ReadTotalTimeoutConstant=10000;  // wait a maximum of 10,000ms = ms


	fSuccess=SetCommTimeouts(
      serial_port, // HANDLE  hFile,	// handle of communications device
      &timeouts    // LPCOMMTIMEOUTS  lpCommTimeouts	// address of communications time-out structure
    );
	if (!fSuccess) {
		/* Handle the error. */
		printf("Can't set Comm Timeout\n");
		exit(1);
	}
    
} // configure_serial

static void send_serial(c)
unsigned char c;
{
	LPDWORD n;
	DWORD d;
    BOOL success;

	n= (&d);
	success= WriteFile(
      serial_port,	// handle to file to write to 
	  (LPCVOID) &c,  // LPCVOID  lpBuffer,	// pointer to data to write to file 
	  (DWORD) 1,   // DWORD  nNumberOfBytesToWrite,	// number of bytes to write 
      n,  // LPDWORD  lpNumberOfBytesWritten,	// pointer to number of bytes written 
      NULL // LPOVERLAPPED  lpOverlapped 	// addr. of structure needed for overlapped I/O  
    );
	if (!success) {
		printf("Write %x failed\n",c);
		exit(1);
	}

	//fprintf(serial_port,"%c",c);
	chars_sent++;
} // send_serial

static void flush_serial() {
	BOOL success;

	success=FlushFileBuffers(serial_port);  
	if (!success) {
		printf("Can't flush serial port buffer\n");
		exit(1);
	}
} // flush_serial

// ASCII control characters used by eval. board to acknowledge packets
#define ACK 0x06
#define NAK 0x15
#define CAN 0x18
#define ETX 0x30

static unsigned char get_acknowledgement()
{
	BOOL success;
	LPDWORD eptr;
	LPVOID bptr;
	DWORD d;
	char buff;

	eptr = (LPDWORD) &d;
	bptr = (LPVOID) &buff;
	
	// Because the transmit code sends a SLIP framing character before AND
	// after the packet with perfect transmission there will be a spurious
	// empty packet after every real packet.  If there is noise on the line
	// this spurious packet might appear to the receiver to contain data.
	// Either way the receiver will generate a NAK for this packet.  Need
	// to clear out the input buffer to get rid of any stored up NAK's
	success=PurgeComm(serial_port,PURGE_RXCLEAR);
	if (!success) {
		printf("Error purging input buffer\n");
		exit(1);
	}

	success=WaitCommEvent(
		serial_port, // HANDLE  hFile,	// handle of communications device
        eptr, // LPDWORD  lpEvtMask,	// address of variable for event that occurred  
        NULL       // LPOVERLAPPED  lpOverlapped, 	// address of overlapped structure
    );
	if (!success) {
		printf("Error waiting to receive char\n");
		exit(1);
	}
	success=ReadFile(
		serial_port, // HANDLE  hFile,	// handle of file to read 
        bptr,        // LPVOID  lpBuffer,	// address of buffer that receives data  
        1,   // DWORD  nNumberOfBytesToRead,	// number of bytes to read 
        eptr, // LPDWORD  lpNumberOfBytesRead,	// address of number of bytes read 
        NULL //LPOVERLAPPED  lpOverlapped 	// address of structure for data 
   );
	if (!success) {
		printf("Error reading char\n");
		exit(1);
	}

	if (buff==ETX) {
		printf("Got ETX\n");
		download_status=DOWNLOAD_SUCCEEDS;
		return 2;
	} else if (buff==ACK) {
		printf("Got ACK\n");
		return 1;
    } else if (buff==NAK) {
		printf("Got NAK\n");
		return 0;
	} else if (buff==CAN) {
	// CAN indicates that the MAC check on the whole download has failed, however
	// the last packet was delivered OK.  So retransmitting the last packet is
	// not appropriate.

		printf("Download Failed\n");
		download_status=DOWNLOAD_FAILS;
		return 1;
	}
	printf("Got %x\n",buff);
	// Its a spurious character - retransmitting the packet is a good idea.
    return 0;
} // get_acknowledgement

//------------------------- END PHYSICAL LAYER CODE ------------------

#define SLIP_ESC 0xdb
#define SLIP_END 0xc0
#define SLIP_ESC_SUB 0xdd
#define SLIP_END_SUB 0xdc



static void slip_send(c)
unsigned char c;
{
   if (c==SLIP_ESC) {
	   send_serial(SLIP_ESC);
	   send_serial(SLIP_ESC_SUB);
   }
   else if (c==SLIP_END) {
	   send_serial(SLIP_ESC);
	   send_serial(SLIP_END_SUB);
   }
   else send_serial(c);
} // slip send

static void slip_start_datagram()
{
	//printf("Slip start datagram\n");
	send_serial(SLIP_END);
} // slip_start_datagram

static void slip_end_datagram()
{
    //printf("Slip end datagram\n");
	send_serial(SLIP_END);
	flush_serial();  
	// Its important that the data is sent NOW because we will want to
	// wait for the acknowledgement.
	printf("Chars Sent=%x\n",chars_sent);
} // slip_end_datagram


//-------------------------- END OF SLIP FRAMING CODE --------------------

// Datagram = (Length, CRC, up to 256 bytes of data)
// Allow a 2 byte field for datagram length for flexibility  (currently
// are just exceeding one byte limit at 258 entries).
#define DATAGRAM_LENGTH 0
#define DATAGRAM_CHECKSUM 2
#define DATAGRAM_DATA 3
#define DATAGRAM_SIZE 258

static unsigned char datagram[DATAGRAM_SIZE];
static unsigned short datagram_index;

#define MAX_ATTEMPTS 3

static void send_datagram() {
	unsigned char checksum;
	int i,attempt;

	datagrams_sent++;
	printf("Send Datagram %d\n",datagrams_sent);
	for (attempt=0;attempt<MAX_ATTEMPTS;attempt++)
	{
		slip_start_datagram();
		// Two byte field for datagram size so the receiving software knows when
		// its finished.
		datagram[DATAGRAM_LENGTH]=(unsigned char)(datagram_index>>8);
		datagram[DATAGRAM_LENGTH+1]=(unsigned char)(datagram_index&0xFF);
		// Simple 8 bit checksum to detect corruption on the data which requires
		// the datagram to be retransmitted.
		checksum=0;
		for (i=DATAGRAM_DATA;i<datagram_index;i++) {
			checksum+=datagram[i];
		}
		datagram[DATAGRAM_CHECKSUM]=checksum;
		// The actual data
		for (i=0;i<datagram_index;i++) {
			slip_send(datagram[i]);
		}
		slip_end_datagram();
		// Wait for acknowledgement before transmitting anything else.
		// Should really have a timeout on the acknowledgement.
		if (get_acknowledgement()) {
			datagram_index=DATAGRAM_DATA;
			return;
		}
		// datagram was not acknowledged - retransmit.
	}
	printf("Failed to send datagram after %d attempts\n",MAX_ATTEMPTS);
	exit(1);
} // send_datagram

void send_char(c)
unsigned char c;
{
   datagram[datagram_index]=c;
   datagram_index++;
   if (datagram_index==DATAGRAM_SIZE) {
	   send_datagram();
   }
} // send_char

void send_long(l)
unsigned long l;
{
  int i;
  for (i=0;i<4;i++) {
     send_char ((unsigned char) ((l>>24)&0xFF));  // Get most sig byte in least sig. byte position and send
	 l=(l<<8);     // Get next most sig. byte ready to send
  }
}  // slip_send_long

//-------------------------- END DATAGRAM CODE ---------------------------

void setup_channel (void) {

	configure_serial();
	datagram_index=DATAGRAM_DATA;
	download_status=IN_PROGRESS;
	chars_sent=0;
} /* setup channel */

void close_channel (void) {
	if (datagram_index>DATAGRAM_DATA) send_datagram();
	
} // close channel

