/*------------------------------------------------------------------------------

  readhex.c

  (c) Copyright 1999 by Triscend Corporation (www.triscend.com).
  All Rights Reserved.

  This code reads in an Intel HEX format file representing a design for the
  Triscend CPSU and processes it to determine the FLASH memory contents.

------------------------------------------------------------------------------*/



#include <stdlib.h>                /* prototype declarations for I/O functions */
#include <stdio.h>
#include <ctype.h>

#include "readhex.h"

/* Array to represent 128K bytes of FLASH memory which will hold the downloaded
   program image. */
unsigned char flash[131072];
/* Number of bytes in the flash array that contain program data */
unsigned long flash_image_size;

// Since memory is 128K bytes the MS word of this address is either 0 or 1.
static unsigned long address_high;

// Need a couple of little routines to convert ASCII representation of hex digits
// into numbers.  The libraries that come with the Microsoft compiler are not up to
// the job because they need spaces between the numbers to synchronise.
static unsigned char hex_to_nybble(f)
FILE *f;
{
	unsigned char c;
	fscanf(f,"%c",&c);
	if (isdigit(c)) {c-='0';} 
	else if (isxdigit(c)) {
		c= ( c-((isupper(c))?'A':'a' ) )+10; // so 'A' hex = 10
	} else {
        fprintf(stderr,"Fatal Error - Invalid Hex digit %c.\n",c);
		exit(1);
	}
    return c;
} // hex_to_nybble

static unsigned char get_byte(f)
FILE *f;
{
   return ((hex_to_nybble(f)<<4) + hex_to_nybble(f));
} // get_byte

static unsigned short get_short(f)
FILE *f;
{
   return ((get_byte(f)<<8) + get_byte(f));
} // get_short


/* See the description of the Intel .hex file format on page 152 of the Keil
   Utlities manual.  The hex files output by Triscend also contain records
   with type field=04, since the entire bitstream does not fit in a single
   64K byte space */
static int read_hex_record(f)
FILE *f;
{
	unsigned short addr;
	unsigned char head,length,type,checksum,c;
	unsigned int i;

	// This code does not verify the checksum on the records since it will
	// read the data from a disk file - thus there is little chance of corruption.
	// Note leading space in scanf format string to skip over newlines and white space.
	fscanf(f," %c",&head);
	if (head!=':') {
		fprintf(stderr,"Fatal Error - Invalid record in hex file. Header=%c\n",head);
		exit(1);
	}
    length=get_byte(f);
	addr=get_short(f);
	type=get_byte(f);
	if (type==1) {
		// End of File Record.
		checksum=get_byte(f);
		printf("End Of Hex File Reached\n");
		return 1;
	} else if (type==4) {
		// Change high address.  Address in each data record is only 16 bits.
		// This record specifies a 16 bit MS word for the subsequent packets.

		if (address_high>1) {
			fprintf(stderr,"Fatal Error - Address outside 128K FLASH memory.\n");
			exit(1);
		}
		address_high=(get_short(f)<<16);  // Top word of a 32 bit address
		checksum=get_byte(f);
	} else if (type==0) {
		// Data packet -- load into memory.
		for (i=0;i<length;i++) {
			c=get_byte(f);
			flash[address_high+addr+i]=c;  // addr is the 16 bit offset for this data block
			if ((address_high+addr+i)>=flash_image_size) {
				flash_image_size=address_high+addr+i+1;
				// +1 so if there is a single byte at address 0 the image size is 1
				// image size==0 implies no code at all.
			}
		}
		checksum=get_byte(f);
    } else {
		fprintf(stderr,"Fatal Error - Invalid HEX file record type %d.\n",type);
	    exit(1);
	}
	return 0;
} /* read_hex_record */

void process_hex_file(filename)
char filename[];
{
   FILE *f=fopen(filename,"r");
   int end_of_file;
   flash_image_size=0;

   for (end_of_file=0; !end_of_file; end_of_file=read_hex_record(f));
   
   printf("Flash Image Size=%x\n",flash_image_size);
} // process_hex_file

